;************************************************************************
;                                                                       *
;   Filename:       XmasStar.asm                                        *
;   Date:           18/10/08                                            *
;   File Version:   2.1                                                 *
;                   (relocatable object version)                        *
;                                                                       *
;   Author:         David Meiklejohn                                    *
;   Company:        Gooligum Electronics                                *
;                                                                       *
;                                                                       *
;************************************************************************
;                                                                       *
;   Architecture:   Midrange PIC                                        *
;   Processor:      12F683                                              *
;                                                                       *
;                                                                       *
;************************************************************************
;                                                                       *
;   Christmas Star lights display                                       *
;                                                                       *
;   Display successive patterns looked up from tables,                  *
;   driven by timer-based interrupt                                     *
;                                                                       *
;   Sequence held in EEPROM:                                            *
;   Each byte in EEPROM represents a pattern of up to 4 LEDs,           *
;   or a command to change display speed,                               *
;   set up a repeating sequence (pattern loop),                         *
;   or to "twinkle" at different rates                                  *
;                                                                       *
;   Lookup tables are used to translate to unique PORT/TRIS values      *
;   which will light individual LEDs (charlieplexing)                   *
;                                                                       *
;   Sleeps (turned off) until button is pressed                         *
;                                                                       *
;   Shuts down on button press or after timeout                         *
;                                                                       *
;************************************************************************
;                                                                       *
;   Pin assignments:                                                    *
;       GP0,1,2,4,5 - LEDs (20, charlieplexed)                          *
;       GP3         - pushbutton (active low, external pull-up)         *
;                                                                       *
;************************************************************************

    list        p=12f683    
    #include    <p12f683.inc>  
    
    errorlevel  -302            ; no "register not in bank 0" warnings
    
    radix       dec
    

;***** CONFIGURATION
                ; fail-safe clock monitor enabled,
                ; internal external switchover mode disabled,
                ; code and data protection off, no brownout detect, int reset,
                ; no watchdog, power-up timer on, 4 MHz int clock
    __CONFIG    _FCMEN_ON & _IESO_OFF & _CP_OFF & _CPD_OFF & _BOD_OFF & _MCLRE_OFF & _WDT_OFF & _PWRTE_ON & _INTRC_OSC_NOCLKOUT 


;***** CONSTANTS
    constant DEFSPEED=184   ; Initial value of display rate:
                            ; each pattern displayed for 2048*(256-speed) us
                            ; (higher is faster)
                            ; speed=184 -> 0.147s -> 6.8Hz

    constant TIMEOUT=10     ; shutdown period = TIMEOUT * 17.9 min
                            ; 10 -> shutdown after 179 min (3.0 hrs)


;***** VARIABLE DEFINITIONS
GENVAR  UDATA
rand        res 1           ; 8-bit random number 
tr8         res 1           ; temp used in rand routine
db_cnt      res 1           ; debounce count
d1          res 1           ; delay counters
d2          res 1
        
SHVAR   UDATA_SHR
index       res 1           ; index into pattern sequence (EEPROM)
pattern     res 1           ; index into pattern tables
led         res 1           ; index into PORT/TRIS tables
speed       res 1           ; display rate: pattern time = (256-speed)*2.05ms
tmcnt       res 3           ; 24-bit (LE) timeout counter
lpstart     res 1           ; start (index position) of current pattern loop
lpcount     res 1           ; pattern loop counter
msval       res 1           ; passed to variable delay routine: #ms to delay
w_temp      res 1           ; context saving 
status_temp res 1
pclath_temp res 1
        
        
;**********************************************************************
RESET   CODE    0x000           ; processor reset vector
        goto    Start           ; go to beginning of program


;***** INTERRUPT SERVICE ROUTINE
INTRPT  CODE    0x004           ; interrupt vector
        movwf   w_temp          ; save context: W, STATUS and PCLATH
        movf    STATUS,w    
        movwf   status_temp    
        movf    PCLATH,w    
        movwf   pclath_temp    
        ; what type of interrupt is this?
        btfsc   INTCON,T0IF     ; is it timer0?
        goto    Timer0Int
        banksel PIR1
        btfsc   PIR1,TMR1IF     ; is it timer1?
        goto    Timer1Int
                                ; if not, must be an IO change interrupt
                                
IOChangeInt                     ; handle IO change interrupt (called on wake from sleep)
        banksel GPIO
        movf    GPIO,w          ; clear mismatch condition by reading input
        bcf     INTCON,GPIF     ; clear IO interrupt flag on completeion
        goto    IntDone        
        
Timer0Int                       ; handle timer0 interrupt: increment timeout count    
        incf    tmcnt,f         ; TMR0 increments every 64us
        btfsc   STATUS,Z        ; -> ++tmcnt (LSB) every 256*64us = 16.4ms
        incf    tmcnt+1,f       ; -> ++tmcnt+1 every 256*16.4ms = 4.19s
        btfsc   STATUS,Z
        incf    tmcnt+2,f       ; -> ++tmcnt+2 (MSB) every 256*4.19s = 1070s = 17.9min
        bcf     INTCON,T0IF     ; clear timer0 interrupt flag on completion
        goto    IntDone
        
Timer1Int                       ; handle timer1 interrupt: increment pattern index
        incf    index,f         ; increment pattern lookup index
        banksel T1CON
        bcf     T1CON,TMR1ON    ; turn off timer before setting TMR1H = speed
        movf    speed,w         ; TMR1L increments every 8us
        movwf   TMR1H           ; -> TMR1H increments every 256*8us = 2048us
        clrf    TMR1L           ; -> next overflow at (256-speed)*2.05ms
        bsf     T1CON,TMR1ON    ; then turn back on
        banksel PIR1
        bcf     PIR1,TMR1IF     ; clear timer1 interrupt flag on completion

IntDone
        movf    pclath_temp,w   ; restore context: PCLATH, STATUS and W
        movwf   PCLATH            
        movf    status_temp,w    
        movwf   STATUS        
        swapf   w_temp,f        ; (restore W without affecting Z flag)
        swapf   w_temp,w
        retfie    


;***** MAIN PROGRAM
MAIN    CODE
Start   ; initialisation
        ; configure timer0
        banksel TMR0
        clrf    TMR0            ; initialise: TMR0 = 0
        movlw   b'11010101'     ; configure Timer0:
        banksel OPTION_REG      ; instruction clock/64 (64us/tick) and turned on
        movwf   OPTION_REG
        ; configure timer1
        movlw   b'00110001'     ; configure Timer1:
        banksel T1CON           ; instruction clock/8 (8us/tick) and turned on
        movwf   T1CON
        ; configure interrupts
        banksel PIE1
        bsf     PIE1,TMR1IE     ; enable timer1 interrupts    
        banksel IOC
        bsf     IOC,3           ; enable interrupt on change on GP3 (for wakeup)
        ; enable interrupts
        movlw   b'11101000'     ; enable peripheral, timer0, GPIO and global interrupts
        movwf   INTCON    

Restart                         ; Go into standby, sleep, then start
        ; reset ports
        movlw   b'111111'       ; turn off lights    
        banksel TRISIO          ; by making all I/O lines inputs
        movwf   TRISIO
        ; go to sleep
        sleep                   ; sleep until button is pressed
        call    db_up           ; debounce switch (wait for stable up)
        
        ; set default speed and initialise timer1
        banksel TMR1L
        clrf    TMR1L
        movlw   DEFSPEED        ; Set initial pattern display rate to DEFSPEED
        movwf   speed           ; and load into timer1
        movwf   TMR1H           ; [will overflow in (256-DEFSPEED)*2.05ms]
        ; initialise variables
        clrf    index           ; start with first lookup pattern (0)
        clrf    tmcnt           ; clear timeout count
        clrf    tmcnt+1
        clrf    tmcnt+2
        movlw   0x45            ; set initial random seed
        banksel rand
        movwf   rand

MainLoop
        ; get pattern from EEPROM
        movf    index,w         ; use index as EEPROM address to read
        banksel EEADR
        movwf   EEADR
        bsf     EECON1,RD       ; initiate read
        movf    EEDATA,w
        movwf   pattern         ; pattern = byte read from EEPROM
        
        ; decode pattern
        btfss   pattern,7       ; pattern with bit 7 high indicates loop or speed change
        goto    chkeot
        btfsc   pattern,6       ; bit 7 + bit 6 high indicates speed change
        goto    setspd
        goto    patloop         ; else (bit 7 high, bit 6 low) pattern loop
chkeot  movlw   0x7F            ; 0x7F indicates end of table
        xorwf   pattern,w
        btfsc   STATUS,Z
        goto    endtbl        
        movlw   .92             ; >=92 indicates twinkling (fast random patterns)
        subwf   pattern,w       ; W = pattern - 92
        btfsc   STATUS,C        ; (twinkle if pattern - 92 is +ve)
        goto    twinkle
        goto    setleds         ; otherwise it's a normal pattern lookup
        
        ; speed change
setspd  bcf     STATUS,C        ; new speed = lower 6 bits of pattern * 4
        rlf     pattern,f       ; -> rotate left with clear carry
        bcf     STATUS,C        ;    twice
        rlf     pattern,w
        movwf   speed           ; set timer1 to new speed
        banksel T1CON
        bcf     T1CON,TMR1ON    ; (turn off timer first)
        movwf   TMR1H           ; (TMR1H = new speed)
        clrf    TMR1L        
        bsf     T1CON,TMR1ON    ; (then turn back on)    
        incf    index,f         ; go immediately to next pattern            
        goto    cktmout         ; restart loop    
        
        ; pattern loop
patloop movlw   b'01111111'     ; count = lower 7 bits of pattern (pattern - 128)
        andwf   pattern,w
        btfsc   STATUS,Z        ; if count = 0, this is an end of loop marker
        goto    endloop        
        movwf   lpcount         ; otherwise set the loop count
        incf    index,w         ; set start of loop (next pattern)
        movwf   lpstart
        movwf   index           ; skip straight to it
        goto    cktmout         ; restart loop
        
endloop movf    lpstart,w       ; get loop start in case loop is not finished
        incf    index,f         ; increment index in case loop is finished
        decfsz  lpcount,f       ; decrement loop counter
        movwf   index           ; if not zero yet, reset index to start of loop
        goto    cktmout         ; restart loop
        
        ; end of table
endtbl  clrf    index           ; set index back to 0
        movlw   DEFSPEED        ; and speed back to default
        movwf   speed
        goto    cktmout         ; restart loop
        
        ; twinkle (fast random patterns)
        ; rate (ms per twinkle change) set by pattern - 91
twinkle movwf   msval           ; at this point W = pattern - 92, so store
        incf    msval,f         ; then increment it to get pattern - .1  
        call    Random8         ; generate next random number, using last as seed
        movlw   b'00011111'     ; W = lower 5 bits of rand
        banksel rand
        andwf   rand,w
        addlw   .235            ; reduce W to range 0 - 20:
        btfss   STATUS,C        ;     W = W - .1  (235 = -21)
        addlw   .21             ;     if result is -ve (C-0), undo the subtraction
        call    setLED          ; turn on corresponding LED
        call    dlyms           ; for msval ms
        goto    cktmout
        
        ; multiplex leds for pattern read from EEPROM
setleds call    getLED1         ; lookup 1st LED in pattern
        call    setLED          ; and turn it on
        call    dly200u         ; delay so that duty cycle is even
        call    getLED2         ; multiplex with 2nd
        call    setLED
        call    dly200u
        call    getLED3         ; then 3rd
        call    setLED    
        call    dly200u
        call    getLED4         ; and 4th
        call    setLED
        call    dly200u

        ; check for timeout then restart loop
cktmout movf    tmcnt+2,w
        xorlw   TIMEOUT         ; if timer count MSB has reached timeout
        btfsc   STATUS,Z        ; go into standby
        goto    Restart
        banksel GPIO
        btfsc   GPIO,3          ; if button not pressed (=0)
        goto    MainLoop        ; repeat forever
        clrf    GPIO            ; else switch was pressed so turn off LEDs,
        call    db_up           ; debounce switch (wait for stable up)
        
        goto    Restart         ; then go into standby


;***** SUBROUTINES
SUBS    CODE

; Delay routines
dly200u         ; delay 200us
        movlw   .65
        banksel d1
        movwf   d1
d200ul  decfsz  d1,f            ; (delay = 2 + 2 + 65 * 3 - 1 + 2 = 200 uS)
        goto    d200ul
        return


dlyms           ; variable delay - ~#ms passed in msval
        movf    msval,w
        banksel d2
        movwf   d2
        clrf    d1
d5ml    nop                     ; (delay is 5 * (4 * 256 + 3) uS)
        incfsz  d1,f
        goto    d5ml        
        decfsz  d2,f
        goto    d5ml        
        return        


; Debounce switch (wait for switch up for 10 ms)
db_up   ; wait until button released (GP3 high), debounce by counting:
        movlw   .13             ; max count = 10ms/768us = 13
        banksel db_cnt
        movwf   db_cnt        
        clrf    d1             
up_dly  incfsz  d1,f            ; delay 256x3 = 768us.
        goto    up_dly
        banksel GPIO
        btfss   GPIO,GP3        ; if button down (GP3 clear),
        goto    db_up           ;   restart count
        banksel db_cnt
        decfsz  db_cnt,f        ; else repeat until max count reached
        goto    up_dly
        return
        

; Pseudo random number generator, using linear shift register feedback
Random8                         ; seed (non-zero) is in rand    
        banksel rand
        movf    rand,w          ; W = R (use to accumulate XOR)
        movwf   tr8             ; T = R (holds shifted result)
        rlf     tr8,f
        rlf     tr8,f           ; T(7) = R(5)
        xorwf   tr8,w           ; W(7) = W(7)^R(5) (= R(7)^R(5))
        rlf     tr8,f           ; T(7) = R(4)
        xorwf   tr8,w           ; W(7) = W(7)^R(4) (= R(7)^R(5)^R(4))
        rlf     tr8,f           ; T(7) = R(3)
        xorwf   tr8,w           ; W(7) = W(7)^R(3) (= R(7)^R(5)^R(4)^R(3))
        addlw   b'10000000'     ; carry = W(7) = R(7)^R(5)^R(4)^R(3)
        rlf     rand,f          ; rotate carry (xor'ed result) into rand
        return                  ; to generate next pseudo-random number in sequence


; Set PORT/TRIS corresponding to given LED
setLED                          ; turn on LED (passed in W)
        banksel GPIO
        clrf    GPIO            ; turn off all before changing to avoid indeterminate values
        movwf   led
        call    getTris         ; lookup tristate value (for led)
        banksel TRISIO
        movwf   TRISIO          ; and set
        call    getPort         ; lookup port value (for led)
        banksel GPIO
        movwf   GPIO            ; and set
        return

getTris                         ; lookup tristate value for LED = led 
        movlw   HIGH TrisV      ; get high order byte of lookup table address
        movwf   PCLATH    
        movlw   LOW TrisV       ; add low byte of table start
        addwf   led,w           ; to led offset and
        movwf   PCL             ; goto computed address

getPort                         ; lookup port value for LED = led
        movlw   HIGH PortV      ; get high order byte of lookup table address
        movwf   PCLATH    
        movlw   LOW PortV       ; add low byte of table start
        addwf   led,w           ; to led offset and
        movwf   PCL             ; goto computed address


; Lookup LEDs corresponding to given pattern
getLED1                         ; lookup LED 1 for this pattern
        movlw   HIGH LED1       ; get high order byte of lookup table address
        movwf   PCLATH          ; table starts on page boundary, so 
        movf    pattern,w       ; low order byte for lookup =  pattern
        movwf   PCL             ; goto computed address

getLED2                         ; lookup LED 2 in same way
        movlw   HIGH LED2
        movwf   PCLATH        
        movf    pattern,w
        movwf   PCL            

getLED3                         ; and LED 3 
        movlw   HIGH LED3    
        movwf   PCLATH        
        movf    pattern,w
        movwf   PCL        

getLED4                         ; and LED 4 
        movlw   HIGH LED4    
        movwf   PCLATH        
        movf    pattern,w
        movwf   PCL        


;***** PORT/TRIS lookup tables
LEDTBL  CODE    0x100
TrisV   ; Tristate values corresponding to each LED
        retlw   b'111111'       ;  0: off
        retlw   b'001111'       ;  1: 5,4
        retlw   b'101011'       ;  2: 4,2
        retlw   b'111001'       ;  3: 2,1
        retlw   b'111100'       ;  4: 1,0
        retlw   b'001111'       ;  5: 4,5
        retlw   b'101011'       ;  6: 2,4
        retlw   b'111001'       ;  7: 1,2
        retlw   b'111100'       ;  8: 0,1
        retlw   b'011011'       ;  9: 5,2
        retlw   b'111010'       ; 10: 2,0
        retlw   b'011011'       ; 11: 2,5
        retlw   b'111010'       ; 12: 0,2
        retlw   b'011101'       ; 13: 5,1
        retlw   b'011101'       ; 14: 1,5
        retlw   b'101101'       ; 15: 4,1
        retlw   b'101101'       ; 16: 1,4
        retlw   b'101110'       ; 17: 4,0
        retlw   b'101110'       ; 18: 0,4
        retlw   b'011110'       ; 19: 5,0
        retlw   b'011110'       ; 20: 0,5

PortV   ; Port values corresponding to each LED
        retlw   b'000000'       ;  0: off
        retlw   b'100000'       ;  1: 5,4
        retlw   b'010000'       ;  2: 4,2
        retlw   b'000100'       ;  3: 2,1
        retlw   b'000010'       ;  4: 1,0
        retlw   b'010000'       ;  5: 4,5
        retlw   b'000100'       ;  6: 2,4
        retlw   b'000010'       ;  7: 1,2
        retlw   b'000001'       ;  8: 0,1
        retlw   b'100000'       ;  9: 5,2
        retlw   b'000100'       ; 10: 2,0
        retlw   b'000100'       ; 11: 2,5
        retlw   b'000001'       ; 12: 0,2
        retlw   b'100000'       ; 13: 5,1
        retlw   b'000010'       ; 14: 1,5
        retlw   b'010000'       ; 15: 4,1
        retlw   b'000010'       ; 16: 1,4
        retlw   b'010000'       ; 17: 4,0
        retlw   b'000001'       ; 18: 0,4
        retlw   b'100000'       ; 19: 5,0
        retlw   b'000001'       ; 20: 0,5


;***** Pattern lookup tables
PAT1    CODE    0x200
LED1    ; LEDs corresponding to each pattern (slot 1)
        retlw   .0              ;  0: off
        retlw   .1              ;  1: L1
        retlw   .2              ;  2: L2
        retlw   .3              ;  3: L3
        retlw   .4              ;  4: L4
        retlw   .5              ;  5: L5
        retlw   .6              ;  6: L6
        retlw   .7              ;  7: L7
        retlw   .8              ;  8: L8
        retlw   .9              ;  9: L9
        retlw   .10             ; 10: L10
        retlw   .11             ; 11: L11
        retlw   .12             ; 12: L12
        retlw   .13             ; 13: L13
        retlw   .14             ; 14: L14
        retlw   .15             ; 15: L15
        retlw   .16             ; 16: L16
        retlw   .17             ; 17: L17
        retlw   .18             ; 18: L18
        retlw   .19             ; 19: L19
        retlw   .20             ; 20: L20
        retlw   .8              ; 21: SE arm
        retlw   .14             ; 22: NE arm
        retlw   .15             ; 23: NW arm
        retlw   .2              ; 24: SW arm
        retlw   .3              ; 25: ring 1 - inner
        retlw   .7              ; 26: ring 2 - inner mid
        retlw   .10             ; 27: ring 3 - outer mid
        retlw   .8              ; 28: ring 4 - outer
        retlw   .12             ; 29: Small points
        retlw   .1              ; 30: N S
        retlw   .4              ; 31: E W
        retlw   .3              ; 32: SE1 NW1
        retlw   .7              ; 33: SE2 NW2
        retlw   .10             ; 34: SE3 NW3
        retlw   .8              ; 35: SE4 NW4
        retlw   .2              ; 36: SW1 NE1
        retlw   .6              ; 37: SW2 NE2
        retlw   .11             ; 38: SW3 NE3
        retlw   .9              ; 39: SW4 NE4
        retlw   .3              ; 40: SE half arm - inner
        retlw   .10             ; 41: SE half arm - outer
        retlw   .14             ; 42: NE half arm - inner
        retlw   .19             ; 43: NE half arm - outer
        retlw   .15             ; 44: NW half arm - inner
        retlw   .5              ; 45: NW half arm - outer
        retlw   .2              ; 46: SW half arm - inner
        retlw   .11             ; 47: SW half arm - outer
        retlw   .3              ; 48: SE NW inner half arms
        retlw   .10             ; 49: SE NW outer half arms
        retlw   .2              ; 50: SW NE inner half arms
        retlw   .11             ; 51: SW NE outer half arms
        retlw   .3              ; 52: SE1 SE3
        retlw   .7              ; 53: SE2 SE4
        retlw   .14             ; 54: NE1 NE3
        retlw   .13             ; 55: NE2 NE4
        retlw   .15             ; 56: NW1 NW3
        retlw   .16             ; 57: NW2 NW4
        retlw   .2              ; 58: SW1 SW3
        retlw   .6              ; 59: SW2 SW4
        retlw   .3              ; 60: SE1 SE3 NW1 NW3
        retlw   .7              ; 61: SE2 SE4 NW2 NW4
        retlw   .14             ; 62: NE1 NE3 SW1 SW3
        retlw   .13             ; 63: NE2 NE4 SW2 SW4
        retlw   .3              ; 64: SE1 SE3 NW2 NW4
        retlw   .7              ; 65: SE2 SE4 NW1 NW3
        retlw   .14             ; 66: NE1 NE3 SW2 SW4
        retlw   .13             ; 67: NE2 NE4 SW1 SW3
        retlw   .3              ; 68: inner and outer SE arm 
        retlw   .14             ; 69: inner and outer NE arm 
        retlw   .15             ; 70: inner and outer NW arm 
        retlw   .2              ; 71: inner and outer SW arm 
        retlw   .3              ; 72: inner and outer SE / NW 
        retlw   .14             ; 73: inner and outer NE / SW 
        retlw   .7              ; 74: middle LEDs SE arm 
        retlw   .13             ; 75: middle LEDs NE arm 
        retlw   .16             ; 76: middle LEDs NW arm 
        retlw   .6              ; 77: middle LEDs SW arm 
        retlw   .7              ; 78: middle LEDs SE / NW 
        retlw   .13             ; 79: middle LEDs NE / SW 
        retlw   .3              ; 80: SE inner NW outer
        retlw   .10             ; 81: SE outer NW inner
        retlw   .14             ; 82: NE inner SW outer
        retlw   .19             ; 83: NE outer SW inner
        retlw   .3              ; 84: SE inner NE outer
        retlw   .3              ; 85: SE inner SW outer
        retlw   .14             ; 86: NE inner SE outer
        retlw   .14             ; 87: NE inner NW outer
        retlw   .15             ; 88: NW inner NE outer
        retlw   .15             ; 89: NW inner SW outer
        retlw   .2              ; 90: SW inner SE outer
        retlw   .2              ; 91: SW inner NW outer

PAT2    CODE    0x300
LED2    ; LEDs corresponding to each pattern (slot 2)
        retlw   .0              ;  0: off
        retlw   .0              ;  1: L1
        retlw   .0              ;  2: L2
        retlw   .0              ;  3: L3
        retlw   .0              ;  4: L4
        retlw   .0              ;  5: L5
        retlw   .0              ;  6: L6
        retlw   .0              ;  7: L7
        retlw   .0              ;  8: L8
        retlw   .0              ;  9: L9
        retlw   .0              ; 10: L10
        retlw   .0              ; 11: L11
        retlw   .0              ; 12: L12
        retlw   .0              ; 13: L13
        retlw   .0              ; 14: L14
        retlw   .0              ; 15: L15
        retlw   .0              ; 16: L16
        retlw   .0              ; 17: L17
        retlw   .0              ; 18: L18
        retlw   .0              ; 19: L19
        retlw   .0              ; 20: L20
        retlw   .10             ; 21: SE arm
        retlw   .13             ; 22: NE arm
        retlw   .16             ; 23: NW arm
        retlw   .6              ; 24: SW arm
        retlw   .14             ; 25: ring 1 - inner
        retlw   .13             ; 26: ring 2 - inner mid
        retlw   .19             ; 27: ring 3 - outer mid
        retlw   .20             ; 28: ring 4 - outer
        retlw   .4              ; 29: Small points
        retlw   .12             ; 30: N S
        retlw   .17             ; 31: E W
        retlw   .15             ; 32: SE1 NW1
        retlw   .16             ; 33: SE2 NW2
        retlw   .5              ; 34: SE3 NW3
        retlw   .18             ; 35: SE4 NW4
        retlw   .14             ; 36: SW1 NE1
        retlw   .13             ; 37: SW2 NE2
        retlw   .19             ; 38: SW3 NE3
        retlw   .20             ; 39: SW4 NE4
        retlw   .7              ; 40: SE half arm - inner
        retlw   .8              ; 41: SE half arm - outer
        retlw   .13             ; 42: NE half arm - inner
        retlw   .20             ; 43: NE half arm - outer
        retlw   .16             ; 44: NW half arm - inner
        retlw   .18             ; 45: NW half arm - outer
        retlw   .6              ; 46: SW half arm - inner
        retlw   .9              ; 47: SW half arm - outer
        retlw   .7              ; 48: SE NW inner half arms
        retlw   .8              ; 49: SE NW outer half arms
        retlw   .6              ; 50: SW NE inner half arms
        retlw   .9              ; 51: SW NE outer half arms
        retlw   .10             ; 52: SE1 SE3
        retlw   .8              ; 53: SE2 SE4
        retlw   .19             ; 54: NE1 NE3
        retlw   .20             ; 55: NE2 NE4
        retlw   .5              ; 56: NW1 NW3
        retlw   .18             ; 57: NW2 NW4
        retlw   .11             ; 58: SW1 SW3
        retlw   .9              ; 59: SW2 SW4
        retlw   .10             ; 60: SE1 SE3 NW1 NW3
        retlw   .8              ; 61: SE2 SE4 NW2 NW4
        retlw   .19             ; 62: NE1 NE3 SW1 SW3
        retlw   .20             ; 63: NE2 NE4 SW2 SW4
        retlw   .10             ; 64: SE1 SE3 NW2 NW4
        retlw   .8              ; 65: SE2 SE4 NW1 NW3
        retlw   .19             ; 66: NE1 NE3 SW2 SW4
        retlw   .20             ; 67: NE2 NE4 SW1 SW3
        retlw   .8              ; 68: inner and outer SE arm 
        retlw   .20             ; 69: inner and outer NE arm 
        retlw   .18             ; 70: inner and outer NW arm 
        retlw   .9              ; 71: inner and outer SW arm 
        retlw   .8              ; 72: inner and outer SE / NW 
        retlw   .20             ; 73: inner and outer NE / SW 
        retlw   .10             ; 74: middle LEDs SE arm 
        retlw   .19             ; 75: middle LEDs NE arm 
        retlw   .5              ; 76: middle LEDs NW arm 
        retlw   .11             ; 77: middle LEDs SW arm 
        retlw   .10             ; 78: middle LEDs SE / NW 
        retlw   .19             ; 79: middle LEDs NE / SW 
        retlw   .7              ; 80: SE inner NW outer
        retlw   .8              ; 81: SE outer NW inner
        retlw   .13             ; 82: NE inner SW outer
        retlw   .20             ; 83: NE outer SW inner
        retlw   .7              ; 84: SE inner NE outer
        retlw   .7              ; 85: SE inner SW outer
        retlw   .13             ; 86: NE inner SE outer
        retlw   .13             ; 87: NE inner NW outer
        retlw   .16             ; 88: NW inner NE outer
        retlw   .16             ; 89: NW inner SW outer
        retlw   .6              ; 90: SW inner SE outer
        retlw   .6              ; 91: SW inner NW outer

PAT3    CODE    0x400
LED3    ; LEDs corresponding to each pattern (slot 3)
        retlw   .0              ;  0: off
        retlw   .0              ;  1: L1
        retlw   .0              ;  2: L2
        retlw   .0              ;  3: L3
        retlw   .0              ;  4: L4
        retlw   .0              ;  5: L5
        retlw   .0              ;  6: L6
        retlw   .0              ;  7: L7
        retlw   .0              ;  8: L8
        retlw   .0              ;  9: L9
        retlw   .0              ; 10: L10
        retlw   .0              ; 11: L11
        retlw   .0              ; 12: L12
        retlw   .0              ; 13: L13
        retlw   .0              ; 14: L14
        retlw   .0              ; 15: L15
        retlw   .0              ; 16: L16
        retlw   .0              ; 17: L17
        retlw   .0              ; 18: L18
        retlw   .0              ; 19: L19
        retlw   .0              ; 20: L20
        retlw   .7              ; 21: SE arm
        retlw   .19             ; 22: NE arm
        retlw   .5              ; 23: NW arm
        retlw   .11             ; 24: SW arm
        retlw   .15             ; 25: ring 1 - inner
        retlw   .16             ; 26: ring 2 - inner mid
        retlw   .5              ; 27: ring 3 - outer mid
        retlw   .18             ; 28: ring 4 - outer
        retlw   .1              ; 29: Small points
        retlw   .0              ; 30: N S
        retlw   .0              ; 31: E W
        retlw   .0              ; 32: SE1 NW1
        retlw   .0              ; 33: SE2 NW2
        retlw   .0              ; 34: SE3 NW3
        retlw   .0              ; 35: SE4 NW4
        retlw   .0              ; 36: SW1 NE1
        retlw   .0              ; 37: SW2 NE2
        retlw   .0              ; 38: SW3 NE3
        retlw   .0              ; 39: SW4 NE4
        retlw   .0              ; 40: SE half arm - inner
        retlw   .0              ; 41: SE half arm - outer
        retlw   .0              ; 42: NE half arm - inner
        retlw   .0              ; 43: NE half arm - outer
        retlw   .0              ; 44: NW half arm - inner
        retlw   .0              ; 45: NW half arm - outer
        retlw   .0              ; 46: SW half arm - inner
        retlw   .0              ; 47: SW half arm - outer
        retlw   .15             ; 48: SE NW inner half arms
        retlw   .5              ; 49: SE NW outer half arms
        retlw   .14             ; 50: SW NE inner half arms
        retlw   .19             ; 51: SW NE outer half arms
        retlw   .0              ; 52: SE1 SE3
        retlw   .0              ; 53: SE2 SE4
        retlw   .0              ; 54: NE1 NE3
        retlw   .0              ; 55: NE2 NE4
        retlw   .0              ; 56: NW1 NW3
        retlw   .0              ; 57: NW2 NW4
        retlw   .0              ; 58: SW1 SW3
        retlw   .0              ; 59: SW2 SW4
        retlw   .15             ; 60: SE1 SE3 NW1 NW3
        retlw   .16             ; 61: SE2 SE4 NW2 NW4
        retlw   .2              ; 62: NE1 NE3 SW1 SW3
        retlw   .6              ; 63: NE2 NE4 SW2 SW4
        retlw   .16             ; 64: SE1 SE3 NW2 NW4
        retlw   .15             ; 65: SE2 SE4 NW1 NW3
        retlw   .6              ; 66: NE1 NE3 SW2 SW4
        retlw   .2              ; 67: NE2 NE4 SW1 SW3
        retlw   .0              ; 68: inner and outer SE arm 
        retlw   .0              ; 69: inner and outer NE arm 
        retlw   .0              ; 70: inner and outer NW arm 
        retlw   .0              ; 71: inner and outer SW arm 
        retlw   .15             ; 72: inner and outer SE / NW 
        retlw   .2              ; 73: inner and outer NE / SW 
        retlw   .0              ; 74: middle LEDs SE arm 
        retlw   .0              ; 75: middle LEDs NE arm 
        retlw   .0              ; 76: middle LEDs NW arm 
        retlw   .0              ; 77: middle LEDs SW arm 
        retlw   .16             ; 78: middle LEDs SE / NW 
        retlw   .6              ; 79: middle LEDs NE / SW 
        retlw   .5              ; 80: SE inner NW outer
        retlw   .15             ; 81: SE outer NW inner
        retlw   .11             ; 82: NE inner SW outer
        retlw   .2              ; 83: NE outer SW inner
        retlw   .19             ; 84: SE inner NE outer
        retlw   .11             ; 85: SE inner SW outer
        retlw   .10             ; 86: NE inner SE outer
        retlw   .5              ; 87: NE inner NW outer
        retlw   .19             ; 88: NW inner NE outer
        retlw   .11             ; 89: NW inner SW outer
        retlw   .10             ; 90: SW inner SE outer
        retlw   .5              ; 91: SW inner NW outer

PAT4    CODE    0x500
LED4    ; LEDs corresponding to each pattern (slot 4)
        retlw   .0              ;  0: off
        retlw   .0              ;  1: L1
        retlw   .0              ;  2: L2
        retlw   .0              ;  3: L3
        retlw   .0              ;  4: L4
        retlw   .0              ;  5: L5
        retlw   .0              ;  6: L6
        retlw   .0              ;  7: L7
        retlw   .0              ;  8: L8
        retlw   .0              ;  9: L9
        retlw   .0              ; 10: L10
        retlw   .0              ; 11: L11
        retlw   .0              ; 12: L12
        retlw   .0              ; 13: L13
        retlw   .0              ; 14: L14
        retlw   .0              ; 15: L15
        retlw   .0              ; 16: L16
        retlw   .0              ; 17: L17
        retlw   .0              ; 18: L18
        retlw   .0              ; 19: L19
        retlw   .0              ; 20: L20
        retlw   .3              ; 21: SE arm
        retlw   .20             ; 22: NE arm
        retlw   .18             ; 23: NW arm
        retlw   .9              ; 24: SW arm
        retlw   .2              ; 25: ring 1 - inner
        retlw   .6              ; 26: ring 2 - inner mid
        retlw   .11             ; 27: ring 3 - outer mid
        retlw   .9              ; 28: ring 4 - outer
        retlw   .17             ; 29: Small points
        retlw   .0              ; 30: N S
        retlw   .0              ; 31: E W
        retlw   .0              ; 32: SE1 NW1
        retlw   .0              ; 33: SE2 NW2
        retlw   .0              ; 34: SE3 NW3
        retlw   .0              ; 35: SE4 NW4
        retlw   .0              ; 36: SW1 NE1
        retlw   .0              ; 37: SW2 NE2
        retlw   .0              ; 38: SW3 NE3
        retlw   .0              ; 39: SW4 NE4
        retlw   .0              ; 40: SE half arm - inner
        retlw   .0              ; 41: SE half arm - outer
        retlw   .0              ; 42: NE half arm - inner
        retlw   .0              ; 43: NE half arm - outer
        retlw   .0              ; 44: NW half arm - inner
        retlw   .0              ; 45: NW half arm - outer
        retlw   .0              ; 46: SW half arm - inner
        retlw   .0              ; 47: SW half arm - outer
        retlw   .16             ; 48: SE NW inner half arms
        retlw   .18             ; 49: SE NW outer half arms
        retlw   .13             ; 50: SW NE inner half arms
        retlw   .20             ; 51: SW NE outer half arms
        retlw   .0              ; 52: SE1 SE3
        retlw   .0              ; 53: SE2 SE4
        retlw   .0              ; 54: NE1 NE3
        retlw   .0              ; 55: NE2 NE4
        retlw   .0              ; 56: NW1 NW3
        retlw   .0              ; 57: NW2 NW4
        retlw   .0              ; 58: SW1 SW3
        retlw   .0              ; 59: SW2 SW4
        retlw   .5              ; 60: SE1 SE3 NW1 NW3
        retlw   .18             ; 61: SE2 SE4 NW2 NW4
        retlw   .11             ; 62: NE1 NE3 SW1 SW3
        retlw   .9              ; 63: NE2 NE4 SW2 SW4
        retlw   .18             ; 64: SE1 SE3 NW2 NW4
        retlw   .5              ; 65: SE2 SE4 NW1 NW3
        retlw   .9              ; 66: NE1 NE3 SW2 SW4
        retlw   .11             ; 67: NE2 NE4 SW1 SW3
        retlw   .0              ; 68: inner and outer SE arm 
        retlw   .0              ; 69: inner and outer NE arm 
        retlw   .0              ; 70: inner and outer NW arm 
        retlw   .0              ; 71: inner and outer SW arm 
        retlw   .18             ; 72: inner and outer SE / NW 
        retlw   .9              ; 73: inner and outer NE / SW 
        retlw   .0              ; 74: middle LEDs SE arm 
        retlw   .0              ; 75: middle LEDs NE arm 
        retlw   .0              ; 76: middle LEDs NW arm 
        retlw   .0              ; 77: middle LEDs SW arm 
        retlw   .5              ; 78: middle LEDs SE / NW 
        retlw   .11             ; 79: middle LEDs NE / SW 
        retlw   .18             ; 80: SE inner NW outer
        retlw   .16             ; 81: SE outer NW inner
        retlw   .9              ; 82: NE inner SW outer
        retlw   .6              ; 83: NE outer SW inner
        retlw   .20             ; 84: SE inner NE outer
        retlw   .9              ; 85: SE inner SW outer
        retlw   .8              ; 86: NE inner SE outer
        retlw   .18             ; 87: NE inner NW outer
        retlw   .20             ; 88: NW inner NE outer
        retlw   .9              ; 89: NW inner SW outer
        retlw   .8              ; 90: SW inner SE outer
        retlw   .18             ; 91: SW inner NW outer


;***** EEPROM data

EEPROM  CODE  0x2100            ; pattern sequence

        ; set speed = 8 Hz
        DE      .241            ; speed = 10^6/[8192(256-241)] = 8.1Hz
        
        ; pause
        DE      .0              ; all off
        
        ; half arms acw - 4 times
        DE      0x84            ; count = 4
        DE      .41             ; SE outer
        DE      .43             ; NE outer    
        DE      .45             ; NW outer            
        DE      .47             ; SW outer
        DE      .74             ; SE mid
        DE      .75             ; NE mid    
        DE      .76             ; NW mid    
        DE      .77             ; SW mid
        DE      .40             ; SE inner    
        DE      .42             ; NE inner
        DE      .44             ; NW inner                    
        DE      .46             ; SW inner
        DE      .74             ; SE mid
        DE      .75             ; NE mid    
        DE      .76             ; NW mid    
        DE      .77             ; SW mid
        DE      0x80            ; end loop
        
        
        ; spin whole arms cw - 8 times
        DE      0x88            ; count = 8
        DE      .24             ; SW
        DE      .17             ; W        
        DE      .23             ; NW    
        DE      .1              ; N    
        DE      .22             ; NE
        DE      .4              ; E                                
        DE      .21             ; SE
        DE      .12             ; S
        DE      0x80            ; end loop    
            
        ; set speed = 12Hz
        DE      .246            ; speed = 10^6/[8192(256-246)] = 12.2Hz
        
        ; spin whole arms acw - 12 times
        DE      0x8C            ; count = 12
        DE      .21             ; SE
        DE      .4              ; E
        DE      .22             ; NE
        DE      .1              ; N
        DE      .23             ; NW
        DE      .17             ; W
        DE      .24             ; SW
        DE      .12             ; S
        DE      0x80            ; end loop    

        ; set speed = 16Hz
        DE      .248            ; speed = 10^6/[8192(256-248)] = 15.3Hz
        
        ; spin whole arms cw - 16 times
        DE      0x90            ; count = 16
        DE      .24             ; SW
        DE      .17             ; W        
        DE      .23             ; NW    
        DE      .1              ; N    
        DE      .22             ; NE
        DE      .4              ; E                                
        DE      .21             ; SE
        DE      .12             ; S
        DE      0x80            ; end loop

        ; set speed = 24Hz
        DE      .251            ; speed = 10^6/[8192(256-251)] = 24.4Hz
        
        ; spin whole arms acw - 24 times
        DE      0x98            ; count = 24
        DE      .21             ; SE
        DE      .4              ; E
        DE      .22             ; NE
        DE      .1              ; N
        DE      .23             ; NW
        DE      .17             ; W
        DE      .24             ; SW
        DE      .12             ; S
        DE      0x80            ; end loop            
        
        ; set speed = 8Hz
        DE      .241            ; speed = 10^6/[8192(256-241)] = 8.1Hz
                
        ; flash small points
        DE      .29             ; small points
        DE      .0              ; pause
    
        ; rings out from centre - 8 times
        DE      0x88            ; count = 8
        DE      .25             ; inner
        DE      .26             ; inner mid
        DE      .27             ; outer mid
        DE      .28             ; outer
        DE      0x80            ; end loop    

        ; set speed = 12Hz
        DE      .246            ; speed = 10^6/[8192(256-246)] = 12.2Hz

        ; rings out from centre - 12 times
        DE      0x8C            ; count = 12
        DE      .25             ; inner
        DE      .26             ; inner mid
        DE      .27             ; outer mid
        DE      .28             ; outer
        DE      0x80            ; end loop            
        
        ; set speed = 16Hz
        DE      .248            ; speed = 10^6/[8192(256-248)] = 15.3Hz

        ; rings out from centre - 16 times
        DE      0x90            ; count = 16
        DE      .25             ; inner
        DE      .26             ; inner mid
        DE      .27             ; outer mid
        DE      .28             ; outer
        DE      0x80            ; end loop    

        ; set speed = 20Hz
        DE      .250            ; speed = 10^6/[8192(256-250)] = 20.3Hz

        ; rings out from centre - 20 times
        DE      0x94            ; count = 20
        DE      .25             ; inner
        DE      .26             ; inner mid
        DE      .27             ; outer mid
        DE      .28             ; outer
        DE      0x80            ; end loop    

        ; set speed = 6Hz
        DE      .236            ; speed = 10^6/[8192(256-236)] = 6.1Hz
        
        ; flash small points
        DE      .29             ; small points
        DE      .0              ; pause    
            
        ; complimentary pairs acw - 2 times
        DE      0x82            ; count = 2
        DE      .35             ; SE NW outer
        DE      .31             ; E W
        DE      .39             ; SW NE outer
        DE      .30             ; N S
        DE      .34             ; SE NW outer mid
        DE      .31             ; E W
        DE      .38             ; SW NE outer mid
        DE      .30             ; N S
        DE      .33             ; SE NW inner mid
        DE      .31             ; E W
        DE      .37             ; SW NE inner mid
        DE      .30             ; N S
        DE      .32             ; SE NW inner
        DE      .31             ; E W
        DE      .36             ; SW NE inner
        DE      .30             ; N S
        DE      .33             ; SE NW inner mid
        DE      .31             ; E W
        DE      .37             ; SW NE inner mid
        DE      .30             ; N S
        DE      .34             ; SE NW outer mid
        DE      .31             ; E W
        DE      .38             ; SW NE outer mid
        DE      .30             ; N S
        DE      0x80            ; end loop        
                
        ; spin opposite half arms cw - 5 times
        DE      0x85            ; count = 5
        DE      .51             ; SW NE outer
        DE      .31             ; E W
        DE      .49             ; SE NW outer
        DE      .30             ; N S
        DE      .79             ; SW NE middle
        DE      .31             ; E W
        DE      .78             ; SE NW middle
        DE      .30             ; N S
        DE      .50             ; SW NE inner
        DE      .31             ; E W
        DE      .48             ; SE NW inner
        DE      .30             ; N S
        DE      .79             ; SW NE middle
        DE      .31             ; E W
        DE      .78             ; SE NW middle
        DE      .30             ; N S        
        DE      0x80            ; end loop

        ; flash small points
        DE      .29             ; small points
        DE      .0              ; pause    
        
        ; set speed = 12Hz
        DE      .246            ; speed = 10^6/[8192(256-246)] = 12.2Hz        

        ; chaser - 6 times
        DE      0x86            ; count = 6
        DE      .8              ; SE outer
        DE      .10             ; SE outer mid
        DE      .7              ; SE inner mid
        DE      .3              ; SE inner
        DE      .15             ; NW inner
        DE      .16             ; NW inner mid
        DE      .5              ; NW outer mid
        DE      .18             ; NW outer
        DE      .5              ; NW outer mid
        DE      .16             ; NW inner mid
        DE      .15             ; NW inner
        DE      .3              ; SE inner
        DE      .7              ; SE inner mid
        DE      .10             ; SE outer mid
        DE      .8              ; SE outer
        DE      .9              ; SW outer
        DE      .11             ; SW outer mid
        DE      .6              ; SW inner mid
        DE      .2              ; SW inner
        DE      .14             ; NE inner
        DE      .13             ; NE inner mid
        DE      .19             ; NE outer mid
        DE      .20             ; NE outer
        DE      .19             ; NE outer mid
        DE      .13             ; NE inner mid
        DE      .14             ; NE inner
        DE      .2              ; SW inner
        DE      .6              ; SW inner mid
        DE      .11             ; SW outer mid
        DE      .9              ; SW outer
        DE      0x80            ; end loop

        ; slow for twinkle changes
        DE      .195            ; speed = 10^6/[8192(256-195)] = 2.0Hz
        
        ; pause
        DE      .0              ; all off
        
        ; twinkle patterns
        DE      0x84            ; repeat 4 x
        DE      .126            ; start slowest (35ms)
        DE      .124            ; get faster - 34ms
        DE      .122            ; etc.
        DE      .120
        DE      .118
        DE      .116
        DE      .115
        DE      .114
        DE      .113
        DE      .112
        DE      .111
        DE      .110
        DE      .109
        DE      .108
        DE      .107
        DE      .106
        DE      .105
        DE      .104
        DE      .103
        DE      .102
        DE      .101
        DE      .100
        DE      .99
        DE      .99
        DE      .98
        DE      .98
        DE      .97
        DE      .97
        DE      .96
        DE      .96
        DE      .95
        DE      .95
        DE      .94
        DE      .94
        DE      .93
        DE      .93
        DE      .92             ; max speed - 1ms
        DE      .92
        DE      .92             ; slow down again
        DE      .93
        DE      .93
        DE      .94
        DE      .94
        DE      .95
        DE      .95
        DE      .96
        DE      .96
        DE      .97
        DE      .97
        DE      .98
        DE      .98
        DE      .99
        DE      .99
        DE      .100
        DE      .101
        DE      .102
        DE      .103
        DE      .104
        DE      .105
        DE      .106
        DE      .107
        DE      .108
        DE      .109
        DE      .110
        DE      .111
        DE      .112
        DE      .113
        DE      .114
        DE      .115
        DE      .116
        DE      .118
        DE      .120
        DE      .122
        DE      .124
        DE      .126
        DE      0x80            ; end loop

        
        ; repeat from beginning
        DE      0x7F            ; sequence end


        END
