
#include <ctype.h>
#include <string.h>
#include "config.h"
#include "playback.h"
#include "infrared.h"
#include "ir_codes.h"
#include "sequencer.h"
#include "mains_phase.h"


extern int strncasecmp(const char* s1, const char* s2, size_t n);

static unsigned char this_token_is(char* buf, unsigned short* buf_pos, unsigned short* buf_len, const char* token) {
	unsigned short toklen = strlen(token);
	if( *buf_len-*buf_pos < toklen || strncasecmp(buf+*buf_pos, token, toklen) )
		return 0;
	*buf_pos += toklen;
	return 1;
}
static unsigned char skip_space(char* buf, unsigned short* buf_pos, unsigned short* buf_len) {
	if( *buf_pos == *buf_len || !isspace(buf[*buf_pos]) )
		return 0;
	while( *buf_pos < *buf_len && isspace(buf[*buf_pos]) && buf[*buf_pos] != '\n' )
		++*buf_pos;
	return 1;
}
static unsigned char next_token_is(char* buf, unsigned short* buf_pos, unsigned short* buf_len, const char* token) {
	unsigned short pos;
	if( !isspace(buf[*buf_pos]) )
		return 0;
	pos = *buf_pos;
	skip_space(buf, &pos, buf_len);
	if( this_token_is(buf, &pos, buf_len, token) ) {
		*buf_pos = pos;
		return 1;
	} else {
		return 0;
    }   
}
static unsigned char next_token_is_spaceopt(char* buf, unsigned short* buf_pos, unsigned short* buf_len, const char* token) {
	unsigned short pos;
	pos = *buf_pos;
	skip_space(buf, &pos, buf_len);
	if( this_token_is(buf, &pos, buf_len, token) ) {
		*buf_pos = pos;
		return 1;
	} else {
		return 0;
    }   
}
static unsigned char get_next_token_boolean(char* buf, unsigned short* buf_pos, unsigned short* buf_len) {
	if( next_token_is(buf, buf_pos, buf_len, "yes") )
		return 1;
	else if( next_token_is(buf, buf_pos, buf_len, "no") )
		return 0;
	else
		return 0xFF;
}
static unsigned char read_integer(char* buf, unsigned short* buf_pos, unsigned short* buf_len, unsigned short* to_here) {
	unsigned short ret = 0;
	if( *buf_len == *buf_pos || !isdigit(buf[*buf_pos]) )
		return 0;
	while( *buf_pos < *buf_len && isdigit(buf[*buf_pos]) ) {
		if( ret > 6553 || (ret == 6553 && buf[*buf_pos] > '5') )
			return 0;
		ret = ret*10 + (buf[*buf_pos]-'0');
		++*buf_pos;
	}
	*to_here = ret;
	return 1;
}
static unsigned char ishexdigit(char c) {
	return (c >= '0' && c <= '9') || (c >= 'a' && c <= 'f') || (c >= 'A' && c <= 'F');
}
static unsigned char read_hex(char* buf, unsigned short* buf_pos, unsigned short* buf_len, unsigned short* to_here) {
	unsigned short ret = 0;
	if( *buf_len == *buf_pos || !ishexdigit(buf[*buf_pos]) )
		return 0;
	while( *buf_pos < *buf_len && ishexdigit(buf[*buf_pos]) ) {
		if( ret > 0xFFF )
			return 0;
		if( buf[*buf_pos] >= 'a' && buf[*buf_pos] <= 'f' ) {
			ret = ret*16 + 10 + (buf[*buf_pos]-'a');
		} else if( buf[*buf_pos] >= 'A' && buf[*buf_pos] <= 'F' ) {
			ret = ret*16 + 10 + (buf[*buf_pos]-'A');
		} else {
			ret = ret*16 + (buf[*buf_pos]-'0');
		}
		++*buf_pos;
	}
	*to_here = ret;
	return 1;
}

extern unsigned char dont_start_playback_automatically, dont_repeat_all, infrared_logging, dc_slaves[4];
extern unsigned short triac_turnoff_delayed[2];
extern signed short zero_crossing_offset;

void reset_config_to_default() {
	controller_phases[0] = controller_phases[1] = controller_phases[2] = controller_phases[3] = 0;
	dc_slaves[0] = dc_slaves[1] = dc_slaves[2] = dc_slaves[3] = 0;
	filament_preheat_off[0] = filament_preheat_off[1] = 0;
	triac_turnoff_delayed[0] = triac_turnoff_delayed[1] = 0;
	filament_preheat_amount = 20;
	set_playback_order(directory);
	dont_start_playback_automatically = 0;
	dont_repeat_all = 0;
	playback_volume_set(100);
	infrared_logging = 0;
	zero_crossing_offset = 0;
	restore_default_remote_codes();
	set_playback_order(sorted);
}

static inline unsigned short abs(signed short val) {
	return val >= 0 ? val : -val;
}

unsigned char read_config(FIL* file) {
  char buf[256];
  unsigned char ret = 1;
  unsigned short buf_pos = 0, buf_len = 0;
  UINT read;

  while(1) {
    if( f_read(file, (BYTE*)buf + buf_len, sizeof(buf)-buf_len, &read) != FR_OK )
	  return 0;
    buf_len += read;
    if( buf_len == 0 )
      break;
	skip_space(buf, &buf_pos, &buf_len);
    if( buf_pos < buf_len ) {
	  if( buf[buf_pos] == '\n' ) {
        ++buf_pos;
      } else {
		if( this_token_is(buf, &buf_pos, &buf_len, "#") || this_token_is(buf, &buf_pos, &buf_len, "//") ) {
			// skip the comment
			while( buf_pos < buf_len && buf[buf_pos] != '\n' )
				++buf_pos;
		} else if( this_token_is(buf, &buf_pos, &buf_len, "phase") ) {
		  unsigned char which_phase, phase;

		  if( !skip_space(buf, &buf_pos, &buf_len) )
            return 0;
          if( buf_pos == buf_len || buf[buf_pos] < '1' || buf[buf_pos] > '4' )
            return 0;
          which_phase = buf[buf_pos] - '1';
          ++buf_pos;
		  if( !next_token_is(buf, &buf_pos, &buf_len, "=") )
            return 0;
		  if( next_token_is(buf, &buf_pos, &buf_len, "0") ) {
            phase = 0;
            ++buf_pos;
		  } else if( next_token_is(buf, &buf_pos, &buf_len, "+120") || next_token_is(buf, &buf_pos, &buf_len, "-240") ) {
            phase = 7;
            buf_pos += 4;
		  } else if( next_token_is(buf, &buf_pos, &buf_len, "-120") || next_token_is(buf, &buf_pos, &buf_len, "+240") ) {
            phase = 21-7;
            buf_pos += 4;
          } else {
            return 0;
          }
		  controller_phases[which_phase] = phase;
        } else if( this_token_is(buf, &buf_pos, &buf_len, "start") ) {
			if( next_token_is(buf, &buf_pos, &buf_len, "playback") ) {
				unsigned char auto_playback;
				if( !next_token_is(buf, &buf_pos, &buf_len, "automatically") )
					return 0;
				if( !next_token_is(buf, &buf_pos, &buf_len, "=") )
					return 0;
				auto_playback = get_next_token_boolean(buf, &buf_pos, &buf_len);
				if( auto_playback == 0xFF )
					return 0;
				dont_start_playback_automatically = !auto_playback;
			} else if( next_token_is(buf, &buf_pos, &buf_len, "file") ) {
				char* name_start;
				if( !next_token_is(buf, &buf_pos, &buf_len, "=") )
					return 0;
				if( !next_token_is(buf, &buf_pos, &buf_len, "\"") )
					return 0;
				name_start = buf+buf_pos;
				while( buf_pos < buf_len && buf[buf_pos] != '"' && buf[buf_pos] != '\n' )
					++buf_pos;
				if( buf_pos == buf_len || buf[buf_pos] != '"' )
					return 0;

				if( ret == 2 )
					return 0;
				if( goto_specific_wav_file(name_start, buf+buf_pos - name_start) == no_wav_files )
					return 1;
				ret = 2;

				++buf_pos;
			} else {
				return 0;
			}
        } else if( this_token_is(buf, &buf_pos, &buf_len, "default") ) {
			if( next_token_is(buf, &buf_pos, &buf_len, "volume") ) {
				unsigned short volume;
				if( !next_token_is(buf, &buf_pos, &buf_len, "=") )
					return 0;
				skip_space(buf, &buf_pos, &buf_len);
				if( !read_integer(buf, &buf_pos, &buf_len, &volume) || volume > 100 )
					return 0;
				if( !this_token_is(buf, &buf_pos, &buf_len, "%") )
					return 0;

				playback_volume_set(volume);
			} else if( next_token_is(buf, &buf_pos, &buf_len, "file") ) {
				playback_order_t order;
				if( !next_token_is(buf, &buf_pos, &buf_len, "order") )
					return 0;
				if( !next_token_is(buf, &buf_pos, &buf_len, "=") )
					return 0;
				if( next_token_is(buf, &buf_pos, &buf_len, "directory") )
					order = directory;
				else if( next_token_is(buf, &buf_pos, &buf_len, "sorted") )
					order = sorted;
				else if( next_token_is(buf, &buf_pos, &buf_len, "shuffle") )
					order = shuffle;
				else
					return 0;

				set_playback_order(order);
			} else if( next_token_is(buf, &buf_pos, &buf_len, "repeat") ) {
				unsigned char repeat_all;
				if( !next_token_is(buf, &buf_pos, &buf_len, "all") )
					return 0;
				if( !next_token_is(buf, &buf_pos, &buf_len, "=") )
					return 0;
				repeat_all = get_next_token_boolean(buf, &buf_pos, &buf_len);
				if( repeat_all == 0xFF )
					return 0;

				dont_repeat_all = !repeat_all;
			} else {
				return 0;
			}
        } else if( this_token_is(buf, &buf_pos, &buf_len, "filament") ) {
			if( !next_token_is(buf, &buf_pos, &buf_len, "preheat") )
				return 0;
			if( next_token_is(buf, &buf_pos, &buf_len, "amount") ) {
				unsigned short preheat_amount;
				if( !next_token_is(buf, &buf_pos, &buf_len, "=") )
					return 0;
				skip_space(buf, &buf_pos, &buf_len);
				if( !read_integer(buf, &buf_pos, &buf_len, &preheat_amount) || preheat_amount > 255 )
					return 0;
				filament_preheat_amount = preheat_amount;
			} else {
				unsigned short group;
				unsigned char which, on;
				skip_space(buf, &buf_pos, &buf_len);
				if( !read_integer(buf, &buf_pos, &buf_len, &group) || group < 1 || group > 4 )
					return 0;
				if( buf_pos < buf_len && buf[buf_pos] == ':' ) {
					++buf_pos;
					unsigned short num;
					if( !read_integer(buf, &buf_pos, &buf_len, &num) || num < 1 || num > 8 )
						return 0;
					which = 1<<(num-1);
				} else {
					which = 0xFF;
				}
				if( !next_token_is(buf, &buf_pos, &buf_len, "=") )
					return 0;
				on = get_next_token_boolean(buf, &buf_pos, &buf_len);
				if( on == 0xFF )
					return 0;

				if( on ) {
					((unsigned char*)filament_preheat_off)[group-1] &= ~which;
				} else {
					((unsigned char*)filament_preheat_off)[group-1] |=  which;
				}
			}
        } else if( this_token_is(buf, &buf_pos, &buf_len, "triac") ) {
			unsigned short group;
			unsigned char which, on;

			if( !next_token_is(buf, &buf_pos, &buf_len, "turnoff") )
				return 0;

			skip_space(buf, &buf_pos, &buf_len);
			if( !read_integer(buf, &buf_pos, &buf_len, &group) || group < 1 || group > 4 )
				return 0;
			if( buf_pos < buf_len && buf[buf_pos] == ':' ) {
				++buf_pos;
				unsigned short num;
				if( !read_integer(buf, &buf_pos, &buf_len, &num) || num < 1 || num > 8 )
					return 0;
				which = 1<<(num-1);
			} else {
				which = 0xFF;
			}
			if( !next_token_is(buf, &buf_pos, &buf_len, "=") )
				return 0;

			if( next_token_is(buf, &buf_pos, &buf_len, "immediate") ) {
				on = 0;
			} else if( next_token_is(buf, &buf_pos, &buf_len, "delayed") ) {
				on = 1;
			} else {
				return 0;
			}

			if( on ) {
				((unsigned char*)triac_turnoff_delayed)[group-1] |=  which;
			} else {
				((unsigned char*)triac_turnoff_delayed)[group-1] &= ~which;
			}
        } else if( this_token_is(buf, &buf_pos, &buf_len, "slave") ) {
			unsigned short group;

			if( !next_token_is(buf, &buf_pos, &buf_len, "type") )
				return 0;

			skip_space(buf, &buf_pos, &buf_len);
			if( !read_integer(buf, &buf_pos, &buf_len, &group) || group < 1 || group > 4 )
				return 0;
			if( !next_token_is(buf, &buf_pos, &buf_len, "=") )
				return 0;

			if( next_token_is(buf, &buf_pos, &buf_len, "DC") ) {
				dc_slaves[group-1] = 1;
			} else if( next_token_is(buf, &buf_pos, &buf_len, "AC") ) {
				dc_slaves[group-1] = 0;
			} else {
				return 0;
			}
        } else if( this_token_is(buf, &buf_pos, &buf_len, "remote") ) {
			unsigned char i, j;
			unsigned char which;
			unsigned short code;

			if( !next_token_is(buf, &buf_pos, &buf_len, "code") )
				return 0;
			skip_space(buf, &buf_pos, &buf_len);
			for( i = 0; i < NUM_REMOTE_FNS; ++i ) {
				if( this_token_is(buf, &buf_pos, &buf_len, remote_codes[i].fn) )
					break;
			}
			if( i == NUM_REMOTE_FNS )
				return 0;
			if( !next_token_is(buf, &buf_pos, &buf_len, "=") )
				return 0;

			memset(remote_codes[i].lower_codes, 0, sizeof(remote_codes[i].lower_codes));
			memset(remote_codes[i].upper_codes, 0, sizeof(remote_codes[i].upper_codes));

			j = 0;
			while( j < NUM_REMOTE_CODES ) {
				if( next_token_is(buf, &buf_pos, &buf_len, "NEC") ) {
					which = IR_NEC>>16;
			 	} else if( next_token_is(buf, &buf_pos, &buf_len, "RC5") ) {
					which = IR_RC5>>16;
				} else {
					return 0;
				}
				if( !next_token_is_spaceopt(buf, &buf_pos, &buf_len, "(") )
					return 0;
				if( next_token_is_spaceopt(buf, &buf_pos, &buf_len, "0x") ) {
					if( !read_hex(buf, &buf_pos, &buf_len, &code) )
						return 0;
				} else {
					if( !read_integer(buf, &buf_pos, &buf_len, &code) )
						return 0;
				}

				remote_codes[i].lower_codes[j] = code;
				remote_codes[i].upper_codes[j] = which;

				if( !next_token_is_spaceopt(buf, &buf_pos, &buf_len, ")") )
					return 0;
				if( ++j == NUM_REMOTE_CODES || !next_token_is_spaceopt(buf, &buf_pos, &buf_len, ",") )
					break;
			}
        } else if( this_token_is(buf, &buf_pos, &buf_len, "infrared") ) {
			if( !next_token_is(buf, &buf_pos, &buf_len, "logging") )
				return 0;
			if( !next_token_is(buf, &buf_pos, &buf_len, "=") )
				return 0;
			if( next_token_is(buf, &buf_pos, &buf_len, "on") ) {
				infrared_logging = 1;
			} else if( next_token_is(buf, &buf_pos, &buf_len, "off") ) {
				infrared_logging = 0;
			} else {
				return 0;
			}
        } else if( this_token_is(buf, &buf_pos, &buf_len, "zero") ) {
			signed char sign = 1;
			unsigned short temp, delay;
			signed short zco;

			if( !next_token_is(buf, &buf_pos, &buf_len, "crossing") )
				return 0;
			if( !next_token_is(buf, &buf_pos, &buf_len, "offset") )
				return 0;
			if( !next_token_is(buf, &buf_pos, &buf_len, "=") )
				return 0;
			if( next_token_is(buf, &buf_pos, &buf_len, "-") )
				sign = -1;
			else
				skip_space(buf, &buf_pos, &buf_len);
			if( !read_integer(buf, &buf_pos, &buf_len, &temp) )
				return 0;
			if( temp > 360 )
				return 0;
			zco = (signed short)temp * sign * 10;
			if( buf_pos < buf_len && buf[buf_pos] == '.' ) {
				++buf_pos;
				if( buf_pos == buf_len || !isdigit(buf[buf_pos]) )
					return 0;
				zco += (buf[buf_pos] - '0') * sign;
				++buf_pos;
			}

			if( zco == 3600 || zco == -3600 )
				zco = 0;

			// convert from tenths of a degree to timer ticks
			delay = __builtin_muluu(abs(zco), timer_phase_length) / 3600;
			if( zco < 0 )
				delay = timer_phase_length - delay;

			zero_crossing_offset = delay;
        } else {
          return 0;
        }
	    if( !next_token_is(buf, &buf_pos, &buf_len, "\n") )
           return 0;
      }
    }
    memmove(buf, buf+buf_pos, buf_len-buf_pos);
    buf_len -= buf_pos;
    buf_pos = 0;
  }

  return ret;
}

unsigned char find_and_read_config() {
	FIL file;
	if( f_open(&file, "0:\\" CONFIG_FILENAME, FA_READ|FA_OPEN_EXISTING) == FR_OK ) {
		return read_config(&file);
	} else {
		return 1;
	}
}
