
#include "targetver.h"
#include "Undo.h"
#include "Sequence.h"
#include "WindowDrawing.h"
#include "Win32Utilities.h"
#include "WAVBrowse.h"
#ifdef _DEBUG
#include <assert.h>
#endif
#include <stdlib.h>
#include <string.h>
#include <new>
#ifdef _DEBUG
#include "WAVBrowse.h"
#endif


struct undo_entry {
	~undo_entry() { delete [] delta; }
	int which, start, old_len, new_len, sample_start, sample_finish;
	unsigned short* delta;
};

struct undo_record {
	undo_record() { num_entries = 0; entries = 0; new_length = -1; }
	~undo_record() { for( int i = 0; i < num_entries; ++i ) delete [] entries[i].delta; free(entries); }
	int num_entries, new_length;
	undo_entry* entries;
	undo_entry* add_entry() { entries = (undo_entry*)realloc(entries, ++num_entries * sizeof(*entries)); return &entries[num_entries-1]; }
};

undo_record* undo_records;
int undo_nesting, num_undo_records, cur_undo_record, cur_undo_index, undo_records_allocated, undo_clean_point, undo_mark_point;
int max_undo_memory = 10485760;

static int CalcUndoMemory(int max_records = -1) {
	if( max_records == -1 )
		max_records = num_undo_records;

	int ret = max_records * sizeof(*undo_records);
	for( int i = 0; i < max_records; ++i ) {
		ret += undo_records[i].num_entries * sizeof(*undo_records[i].entries);
		for( int j = 0; j < undo_records[i].num_entries; ++j ) {
			ret += max(undo_records[i].entries[j].new_len, undo_records[i].entries[j].old_len) * sizeof(unsigned short);
		}
	}
	return ret;
}
static void ReduceUndoMemory(int num_bytes) {
	int num = 0;
	for( ; num < num_undo_records-1 && num < cur_undo_index; ++num ) {
		if( num > 0 && CalcUndoMemory(num) >= num_bytes )
			break;
	}
	if( num > 0 ) {
		for( int i = 0; i < num; ++i )
			undo_records[i].~undo_record();
		memmove(undo_records, undo_records + num, (num_undo_records - num) * sizeof(*undo_records));
		num_undo_records -= num;
		cur_undo_index -= num;
		undo_clean_point -= num;
		undo_mark_point -= num;
		undo_records = (undo_record*)realloc(undo_records, num_undo_records * sizeof(*undo_records));
		undo_records_allocated = num_undo_records;
	}
}

void UndoBegin() {
	if( ++undo_nesting == 1 ) {
#ifdef _DEBUG
		assert(cur_undo_record == 0);
#endif

		if( cur_undo_index < num_undo_records ) {
			while( cur_undo_index < num_undo_records )
				undo_records[--num_undo_records].~undo_record();
		} else {
			if( num_undo_records >= undo_records_allocated ) {
				undo_records_allocated += 16;
				undo_records = (undo_record*)realloc(undo_records, undo_records_allocated*sizeof(*undo_records));
			}
		}
		new (&undo_records[num_undo_records]) undo_record();
		cur_undo_record = ++num_undo_records;
	}
}
void UndoEnd(HWND hWnd) {
	if( --undo_nesting == 0 ) {
#ifdef _DEBUG
		assert(cur_undo_record != 0);
#endif
		++cur_undo_index;
		cur_undo_record = 0;
		if( cur_undo_index == 1 )
			DelayedUpdateDisabledState(hWnd);

		int undo_memory_used = CalcUndoMemory();
		while( undo_memory_used > max_undo_memory )
			ReduceUndoMemory(undo_memory_used - max_undo_memory);
	}
}
void UndoChangingLength(int new_length) {
#ifdef _DEBUG
	assert( undo_nesting && cur_undo_record <= num_undo_records );
#endif
	undo_records[cur_undo_record-1].new_length = new_length;
}

unsigned short* tmp_undo_cmds;
int tmp_undo_len;
void UndoPrepare(int i) {
	tmp_undo_len = sequences[i].used;
	tmp_undo_cmds = new unsigned short[tmp_undo_len];
	memcpy(tmp_undo_cmds, sequences[i].commands, tmp_undo_len*sizeof(unsigned short));
}
void UndoFinish(int i, int sample_start, int sample_finish) {
	int start, finish;
	for( start = 0; start < sequences[i].used && start < tmp_undo_len; ++start )
		if( tmp_undo_cmds[start] != sequences[i].commands[start] )
			break;
	int offset = sequences[i].used - tmp_undo_len;
	for( finish = tmp_undo_len; finish > start; --finish )
		if( tmp_undo_cmds[finish-1] != sequences[i].commands[finish-1+offset] )
			break;
	if( start != finish ) {
#ifdef _DEBUG
		assert(cur_undo_record != 0);
#endif
		undo_entry* pEntry = undo_records[cur_undo_record-1].add_entry();
		pEntry->which = i;
		pEntry->start = start;
		pEntry->old_len = finish-start;
		pEntry->new_len = pEntry->old_len + offset;
		pEntry->delta = new unsigned short[max(pEntry->new_len, pEntry->old_len)];
		memcpy(pEntry->delta, tmp_undo_cmds + start, pEntry->old_len*sizeof(unsigned short));
		pEntry->sample_start = sample_start;
		pEntry->sample_finish = sample_finish;
	}
	delete [] tmp_undo_cmds;
}

static void SwapShorts(unsigned short* a, unsigned short* b, int old_len, int new_len) {
	unsigned short temp;
	while( old_len ) {
		temp = *a;
		*a = *b;
		*b = temp;
		++a;
		++b;
		--old_len;
		--new_len;
	}
	if( new_len > 0 ) {
		while( new_len ) {
			*b = *a;
			++a;
			++b;
			--new_len;
		}
	}
}

void UndoRedo(HWND hWnd, bool bRedraw) {
#ifdef _DEBUG
	assert(undo_nesting == 0);
#endif
	undo_record* pRecord = &undo_records[cur_undo_index];
	if( pRecord->new_length != -1 ) {
		int new_length = pRecord->new_length;
		int old_length = wav_len;
		wav_len = new_length;
		InvalidateSelectionRange(hWnd, min(old_length, new_length), max(old_length, new_length), 0);
		if( selection_start > new_length || selection_finish > new_length )
			SetSelection(hWnd, min(selection_start, new_length), min(selection_finish, new_length));
		pRecord->new_length = old_length;
	}
	int change_start = -1;
	for( int j = 0; j < pRecord->num_entries; ++j ) {
		undo_entry* pEntry = &pRecord->entries[j];
		int i = pEntry->which;
		int old_used = sequences[i].used;
#ifdef _DEBUG
		assert(old_used >= pEntry->start + pEntry->new_len);
#endif
		if( pEntry->old_len > pEntry->new_len ) {
			sequences[i].used += pEntry->old_len - pEntry->new_len;
			if( sequences[i].used > sequences[i].len ) {
				sequences[i].len = sequences[i].used;
				sequences[i].commands = (unsigned short*)realloc(sequences[i].commands, sequences[i].len*sizeof(unsigned short));
			}
			memmove(sequences[i].commands + pEntry->start + pEntry->old_len, sequences[i].commands + pEntry->start + pEntry->new_len, (sequences[i].used - pEntry->start - pEntry->old_len)*sizeof(unsigned short));
		}
		SwapShorts(sequences[i].commands + pEntry->start, pEntry->delta, pEntry->old_len, pEntry->new_len);
		if( pEntry->new_len > pEntry->old_len ) {
			memmove(sequences[i].commands + pEntry->start + pEntry->old_len, sequences[i].commands + pEntry->start + pEntry->new_len, (old_used - pEntry->start - pEntry->new_len)*sizeof(unsigned short));
		}
		if( pEntry->old_len < pEntry->new_len )
			sequences[i].used += pEntry->old_len - pEntry->new_len;
#ifdef _DEBUG
		assert(sequence_get_length_ms(&sequences[i]) == (wav_len + ((wav_sample_rate+999) / 1000 - 1)) * 1000LL / wav_sample_rate);
#endif
		swap(pEntry->old_len, pEntry->new_len);
		if( change_start == -1 || pEntry->sample_start < change_start )
			change_start = pEntry->sample_start;
		if( bRedraw )
			InvalidateLightRange(hWnd, pEntry->sample_start, pEntry->sample_finish, 1<<i);
	}
	if( change_start != -1 )
		MainSequencesChanged(hWnd, change_start, wav_len);
}

void InvalidateLastUndo(HWND hWnd) {
#ifdef _DEBUG
	assert(cur_undo_index < num_undo_records);
#endif
	undo_record* pRecord = &undo_records[cur_undo_index];
	for( int j = 0; j < pRecord->num_entries; ++j ) {
		undo_entry* pEntry = &pRecord->entries[j];
		int i = pEntry->which;
		InvalidateLightRange(hWnd, pEntry->sample_start, pEntry->sample_finish, 1<<i);
	}
}

void Undo(HWND hWnd, bool bRedraw) {
	if( cur_undo_index ) {
		--cur_undo_index;
		UndoRedo(hWnd, bRedraw);
		DelayedUpdateDisabledState(hWnd);
	}
}

void Redo(HWND hWnd) {
	if( cur_undo_index < num_undo_records ) {
		UndoRedo(hWnd, true);
		++cur_undo_index;
		DelayedUpdateDisabledState(hWnd);
	}
}

void UndoReset(HWND hWnd) {
	for( int i = 0; i < num_undo_records; ++i )
		undo_records[i].~undo_record();
	free(undo_records);
	undo_records = 0;
	num_undo_records = 0;
	cur_undo_record = 0;
	cur_undo_index = 0;
	undo_records_allocated = 0;
	undo_clean_point = 0;
	undo_mark_point = 0;
	DelayedUpdateDisabledState(hWnd);
}

bool UndoIsDirty() {
	return cur_undo_index != undo_clean_point;
}

void UndoSetDirty() {
	undo_clean_point = -1;
}

void UndoSetClean() {
	undo_clean_point = cur_undo_index;
}

void MarkUndo() {
	undo_mark_point = cur_undo_index;
}

void UnmarkUndo() {
	undo_mark_point = 0;
}

bool UndoToMark(HWND hWnd, bool bRedraw) {
	if( undo_mark_point > 0 && undo_mark_point <= cur_undo_index ) {
		while( undo_mark_point <= cur_undo_index )
			Undo(hWnd, bRedraw);
		undo_mark_point = 0;
		return true;
	}
	undo_mark_point = 0;
	return false;
}

bool CanUndo(const wchar_t** pReason) {
	if( pReason )
		*pReason = L"there is nothing to undo yet";
	return cur_undo_index != 0;
}

bool CanRedo(const wchar_t** pReason) {
	if( pReason )
		*pReason = L"you can only redo immediately after using the undo function";
	return cur_undo_index < num_undo_records;
}
