// Chess Clock with BV500 
// using 8-digit display with integrated Maxim MAX7219 LED driver
// uses the Rookie V.2 functions for GPIO and SPI 

// V.0 to test the display functions    RSH   30 June 2013
// V.1 adds the logic and tests OK             3 July 2013

// ...............Drive Lines for SPI.................
// RB13		DataOut (Grey)    (DIN)
// RB14		Clock   (White)   (CLK)
// RB15		Load    (Purple)  (LOAD)  NOT a SPI function, needed by MAX7219
// MAX7219 runs on 5V, but accepts inputs at 3V3 levels

// ..................Other I/O assignments..............
// RB0    Master reset - goes low to reset
// RB1    White to play switch    1 is white to play - see notes about flipflop
// RB2    Black to play switch    not used, because RB1 is complement
// RB3    5 Minute countdown switch - 1 is countdown

function init_port() 
  spi_init(50000)
  io_pinMode(PORTB,15,OUT,WOFF)
  io_write(PORTB,15,LOW)  // ensure the LOAD line is low
  io_pinMode(PORTB,0,IN, WPU)   // set the inputs to have a pull-up
  io_pinMode(PORTB,1,IN, WPU)
  io_pinMode(PORTB,2,IN, WPU)
  io_pinMode(PORTB,3,IN, WPU)
endf

// global variables: reset, white_time, black_time, mode, white & black flags,
// white & black switch indicators, white & black flag counters, seconds counts

dim mr, wt, bt, m, wf, bf, w ,b, wfc, bfc, ws, bs

//  mode = 1 is countdown from 5 minutes     
// note that (eg) wt is h.mm when counting up, and m.ss when counting down 

function do_load()
  io_write(PORTB,15,TOGGLE)
  io_write(PORTB,15,TOGGLE)
endf

function send_data(add, value) // sends 2 bytes, then a load pulse
  spi_data(add)
  spi_data(value)
  do_load
endf
 
function init_max()  //setup for MAX7219 
    send_data(9,0x77)// set the decode to Code B for all digits except 4 & 8  
    send_data(10,4)  // set display brighness to quarter (9/32) intensity 
    send_data(11,7)  // set the number of digits to 8
    send_data(12,1)  // make sure the shutdown register is set to normal
endf

function write_w(f,t) // write only 4 leftmost digits
  dim a, b(5), q
  a = t
  for q = 1 to 3
    b(q) = a - (10*(a/10))
    a=a/10
  next q
  send_data(5,b(1))
  send_data(6,b(2))
  send_data (7, (0x80+b(3)))  //  put the decimal point on digit 7
  send_data(8,f)
endf

 function write_b(f,t)  // write only 4 rightmost digits
  dim a, b(5), q
  a = t
  for q = 1 to 3
    b(q) = a - (10*(a/10))
    a=a/10
  next q
  send_data(1,b(1))
  send_data(2,b(2))
  send_data (3, (0x80+b(3)))  //  put the decimal point on digit 3
  send_data(4,f)
endf

function wait_time()    // timer loop, waits 1 second unless reset occurs
  tmr_set(?TIMER23, 10000000)
  mr = 0
  while tmr_get(?TIMER23) < 9999998
    if io_read(PORTB, 0) = 0  // check reset and exit if set
      mr = 1
      return
    endif
  wend
  tmr_clr(?TIMER23)
endf

function inc_w()  // count up white and sort time carry
  ws=ws+1
  if ws=60 
    if wt - (100*(wt/100))=59
      wt = wt+41
    else
      wt = wt+1
    endif
    ws=0
   endif 
endf

function inc_b()  // count up black similarly
    bs=bs+1
    if bs=60
      if bt - (100*(bt/100))=59
        bt=bt+41
      else
        bt=bt+1
      endif
      bs=0
    endif
endf 

function dec_w()  // count down white and sort minute transitions
  if wt=0  // quit if already zero
    return
  endif
  if wt - (100*(wt/100))=0
    wt=wt-41
  else
    wt=wt-1
  endif
endf

function dec_b()   // count down black similarly
  if bt=0  // quit if already zero
    return
  endif
  if bt - (100*(bt/100))=0
    bt=bt-41
  else
    bt=bt-1
  endif
endf

function start() // start up and set initial values
  init_port()
  init_max()
  if io_read(PORTB, 3) = 1
    m = 1   // countdown required
    wt = 500
    bt = 500
   else
    m = 0   // normal count up
    wt = 0
    bt = 0
  endif
  wf = 0    // initialise the other variables
  bf = 0
  ws=0
  bs=0
  wfc=0
  bfc=0
  write_w(wf,wt)
  write_b(bf,bt)    
endf

// code to show "F" is 0x47; code for DP is 0x80

function run()    // called every second, counts appropriately and sorts flags   
                  // uses select to simplify the multiple choice logic
  dim p
  w=io_read(PORTB, 1)
  p=2*m + w // produce 4 cases
// p=0 > count black up; p=1 > count white up; p=2 > count black down, etc 
  select(p)
    case(0); inc_b()
             if (bt-(100*(bt/100))=0)&& (bs=0)   // check for overflow
                bf=0x47                 //set flag
                bfc=10                  //set flag count
             else
              if (bfc=0) && (bf=0x47)   // if flag count is zero, reset flag
                bf=0
              endif 
             endif
             bf=bf^0x80                 // flash dp regardless of flag value
             write_b(bf,bt)
             if bfc<>0                  // if flag on, decrement flag count
                bfc=bfc-1
             endif
    case(1); inc_w()
             if (wt-(100*(wt/100))=0) && (ws=0)   // check for overflow
                wf=0x47                 //set flag
                wfc=10                  //set flag count
             else
              if (wfc=0)&& (wf=0x47)    // if flag count is zero, reset flag
                wf=0
              endif 
             endif
             wf=wf^0x80                 // flash dp regardless of flag value
             write_w(wf,wt)
             if wfc<>0                  // if flag on, decrement flag count
                wfc=wfc-1
             endif
    case(2);  if bf=0x47                // stop decrementing if flag is on
                break
              else       
                dec_b()
              endif
              if bt=0
                bf=0x47
                write_b(bf,bt)
                break                   // if flag is on, game is over
              endif
              bf=bf^0x80
              write_b(bf,bt)
    case(3);  if wf =0x47               // stop decrementing if flag is on
                break
              else
                dec_w()
              endif
              if wt=0
                wf=0x47
                write_w(wf,wt)
                break                   // if flag is on, game is over
              endif
              wf=wf^0x80
              write_w(wf,wt)    
  endselect
endf             
    
function chess()  
  start()  
  while comkey?(2) = 0  // make sure there is an exit for debugging!
    wait_time()
    if mr=1
      start()
    else
      run()
    endif
  wend
endf
              
