;*********************************************************************************
;          
;This program is part of a system to obtain Video images from a Video camera
;mounted in a bird box via a 2.4Ghz wireless channel.
;
;The part of the system at the camera end (called the TX) is controlled by
;commands sent at the other end (called the RX) via a 433Mhz radio channel.
;If the command is received and understood correctly, the TX will then respond by
;repeating the command back to the RX via the audio channel of the TX, this
;provides a crude "handshake" system so that the RX does not send any more
;commands until the TX has responded.
;
;At present, the only commands are to turn the TX system on and off and to
;request the TX system to send battery voltage data to the RX. 
;
;NOTE.  The battery voltage is only present at the A/D input when camera and video
;TX are swiched on and since the TX can only reply to commands via the TX audio 
;channel it can only reply if it is already ON or the command is to turn it ON.
;
;This is the code for the TX system which uses the PIC12F675.
;                                                                     
;*********************************************************************************
;                                                                     
;    Filename:	    BcamTX.asm                                        
;    Date:          31/03/09                                          
;    File Version:                                                   
;                                                                        
;    Files Required: P12F675.INC                                     
;                                                                     
;*********************************************************************************

	list      p=12f675     	;list directive to define processor
	#include <p12f675.inc> 	;processor specific variable definitions

	errorlevel  -302    	;suppress message 302 from list file

	__CONFIG _CP_OFF & _CPD_OFF & _BODEN_OFF & _MCLRE_ON & _WDT_OFF & _PWRTE_ON & _INTRC_OSC_NOCLKOUT 

;'__CONFIG' directive is used to embed configuration word within .asm file.
;The lables following the directive are located in the respective .inc file.
;See data sheet for additional information on configuration word settings.

;*********************RESOURCES USED**********************************************
;Internal 4MHz oscillator
;
;I/O PORTS:
;GP0 = A/D input to monitor battery voltage.
;GP1 = Serial data output.
;GP2 = Serial data input.
;GP3 = used for the MCLR input function (see data sheet page 56 passage 9.3.1).
;GP4 = output used to control the power relay.
;GP5 = Manual "camera on" switch input (for testing).
;
;Ram used for variable storage
;
;Timer 1 used to provide  communication time out protection. 
;
;********* VARIABLE STORAGE (RAM LOCATIONS)***************************************
wtemp   EQU     0x20 	      	;variable used for context saving W 
statemp EQU     0x21 	       	;variable used for context saving PIC STATUS 
inbyte	EQU	0x22		;input byte from control
bitsac	EQU	0x23		;holds bit '1's count
samcnt	EQU	0x24		;holds the bit sample count
ipsflg	EQU	0x25		;input status flags
outbyte	EQU	0x26		;output byte to control
battv	EQU	0x27		;battery voltage read from A/D
dcnt	EQU	0x28		;counter used in time delay loop
gpcnt	EQU	0x29		;general purpose counter
gpcnt2	EQU	0x2A		;another counter
;
;***********************DEFINITIONS***********************************************
#define brdyflg	ipsflg,0	;1 = input byte ready flag
#define busyflg	ipsflg,1	;1 = input is busy (receiving a byte)
#define berflg	ipsflg,2	;1 = input bit error (probably interference)
#define	ferflg	ipsflg,3	;1 = input framing error (lost byte sync)
;
#define serin	GPIO,GP2	;GP2 used for serial input.
;
#define	serout	GPIO,GP1	;GP1 used for serial output
;
#define	powsw	GPIO,GP4	;GP4 '1'= camera and TX power OFF, '0' = ON
;
#define mansw	GPIO,5		;GP5 "0" = manual camera on, "1" remote control only
;
;*********************************************************************************
	ORG     0x000		;processor reset vector
	goto    init 		;go to beginning of program
		
	ORG     0x004	 	;interrupt vector location
	Goto	isr
;*****************initialize I/O etc**********************************************
;these first 4 instructions are required because we use the internal oscillator
;Note this sets the cyle time to 1uSec
init	bsf     STATUS,RP0	;set file register bank to 1 
	call    0x3FF   	;retrieve factory calibration value	
	movwf   OSCCAL   	;update register with factory cal value 
;init IO ports
	bcf     STATUS,RP0	;set file register bank to 0	
	movlw	0x39		;init GPIO latches
	movwf	GPIO		;bits 1 and 2 = low, rest high
	bsf	STATUS,RP0	;sel bank1
	movlw	0x2D		;init GPIO directions
	movwf	TRISIO		;bits 1,4 and = output, rest input
	bcf	OPTION_REG,7	;enable pull up	
	bsf	WPU,5		;for GPIO 5 only
;init A/D
	movlw	0x11
	movwf	ANSEL		;A/D clock = fosc/8, GPIO 0 = analog input
	bcf	STATUS,RP0	;sel bank 0		
	movlw	0x01		;A/D res = Left justified, Ref = vdd, sel chan 0,  		
	movwf	ADCON0		;Conversion complete and A/D = on
; Clear all system variables
	movlw	gpcnt2-wtemp	;= num of locations to be cleared
	movwf	wtemp
	movlw	statemp		;setup pointer
	movwf	FSR
clvars	clrf	INDF		;clear var
	incf	FSR,f		;increment pointer
	decfsz	wtemp,f		;cleared all vars ?
	goto	clvars		;no
;If Manual button is being pressed at this time (Start up) go to manual mode
;which does not use interrupts, so don't enable them
	btfss	mansw		;use manual mode ?
	goto	manual		;yes	
;not man mode so init interrupt sys: ONLY ext int (on GP2 rising edge) and
;timer 1 overflow
	movlw	0xD0
	movwf	INTCON		;set GIE,PEIE (for timer1) and INTE (for ext int)
	bsf	STATUS,RP0	;sel bank1		
	movlw	0x01
	movwf	PIE1		;enable timer1 int
	bcf	STATUS,RP0	;sel bank 0
	goto	run
;***********************THE EXECUTIVE OR RUNNING PART*****************************
;Enter manual mode. This mode will only  be exited if the main power switch is
;turned off or the battery voltage falls below 10v 
manual	call	poweron		;yes turn power on and wait for syten to stabilize                                                                                    
manmd	call	gbattv		;monitor battery voltage
	btfsc	STATUS,C	;battery voltage < 10v ?
	goto	manmd		;no
	bsf	powsw		;yes turn camera off
manend	goto	manend		;and wait here

;Enter remote control mode
run	btfsc	powsw		;is power (camera and TX) ON ?
	goto	pflags		;no check for input
	call	gbattv		;yes, read battery voltage
	btfsc	STATUS,C	;battery voltage <10v ?
	goto	pflags		;no check for input
	movlw	0xE8		;yes, Xmit Battery End warning to controller
	call	sbyte
	bsf	powsw		;turn power off to prevent battery deep discharge
pflags	movf	ipsflg,f	;move file to self to test for 0
	btfsc	STATUS,Z	;any input status flags set ?
	goto	run		;no
;input flag/s set find which one/s
bflgtst	btfsc	busyflg		;busy flag ?
	goto	bflgtst		;yes
	btfsc	brdyflg		;no, byte ready flag ?
	goto	decom		;yes
	movlw	0xE9		;no		
	btfsc	berflg		;bit error ?
	goto	senderr		;yes
	movlw	0xEA		;no, must be framing error
senderr	call	sbyte		;send error report byte to controller
	clrf	ipsflg		;clear all input flags
	goto	run		
;command byte received so decode the command (ISR will have placed the command byte
;in "inbyte") 
decom	bcf	brdyflg		;first clear byte ready flag
	movlw	0xC0
	xorwf	inbyte,w
	btfss	STATUS,Z	;command = turn power OFF ?
	goto	tcpon		;no
	movf	inbyte,w	;yes
	call	sbyte		;acknowldge command
	bsf	powsw		;now turn power OFF
	goto	run
tcpon	movlw	0xC1
	xorwf	inbyte,w
	btfss	STATUS,Z	;cmd = turn power ON ?
	goto	tcdata		;no
;When the TX is off a considerable amount of noise is present on the audio channel
;at the RX end which would cause many false interrupts, so they are not enabled at
;the RX end untill 120 mSec after the RX has sent the command to turn the TX on.
;This allows time for the switch to operate and the signal lines to stabilize.
;for this reason we do not send a reply to the RX untill 0.6Sec has elapsed.
;Turning the power on also causes interference on the command channel at this end
;so we also dissable the external interupt during this delay time
	bcf	INTCON,INTE	;dissable ext int
	call	poweron		;Turn power on and wait 600mSec
	bcf	INTCON,INTF	;clr ext int flag
	bsf	INTCON,INTE	;then re-enable int
	movf	inbyte,w	;now send reply
	call	sbyte
	goto	run
tcdata	movlw	0xBD
	xorwf	inbyte,w
	btfss	STATUS,Z	;cmd = send battery data ?
	goto	comerr		;no = command error 
	movf	inbyte,W	;request for data
	call	sbyte		;respond
	btfsc	powsw		;is power (camera and TX) ON ?	
	goto	run		;no, can't read or send battery voltage value
	call	gbattv		;yes, get value
	movf	battv,w
	call	sbyte		;and send it
	goto	run
comerr	movlw	0xEB		;send command error message
	call	sbyte
	goto	run

;*********SUB TO READ BATTERY VOLTAGE*********************************************
;read the battery voltage, the value is scaled (by a potential divider)so that 11v
;yields an A/D value of 110, 12 v yields 120 and so on
;returns with battery voltage value in file "battv".
;The value is also tested  and the result returned in the carry which will be set
;if the voltage is OK and cleared if the voltage is < 10v.
gbattv	bsf	ADCON0,GO	;start A/D
rdadf	btfsc	ADCON0,GO	;A/D done ?
	goto	rdadf		;no
	movf	ADRESH,w	;yes, get value
	movwf	battv		;and save it
	movlw	0x64		;= 100 decimal
	subwf	battv,w		;test for low voltage
	return

;*********SUB ROUTINE TO SEND A BYTE TO CONTROLLER********************************
;enter with byte in W reg
;The external interrupt is dissabled whilst sending a byte so that the timing is
;not disturbed.
sbyte	movwf	outbyte		;save byte
	btfsc	powsw		;is power on ?
	return			;no, so can't send anything
	bcf	INTCON,INTE	;yes, 1st dissable ext interrupt
	movlw	8
	movwf	gpcnt		;to count 8 bits
sbnxb	rlf	outbyte,f	;rotate bit into carry
	bsf	serout		;bring serial out line high
	call	dly1ms		;delay 1 mSec
	btfss	STATUS,C	;bit = 1 ?
	bcf	serout		;no, bring serial out line low
	call	dly1ms		;delay 1ms
	bcf	serout		;allways bring serial out line low at this time
	call	dly1ms		;1 mS to next bit or end of byte
	decfsz	gpcnt,f		;done 8 bits ?
	goto	sbnxb		;no
	bcf	INTCON,INTF	;yes,make sure flag is clear before enabling int
	bsf	INTCON,INTE	;re- enable ext int	
	return

;**********SUB ROUTINE TO TURN POWER ON AND WAIT 600mSec**************************
;provide this delay by calling dly1ms 600 times.
poweron	bcf	powsw		;turn power ON
	movlw	03		;load outer counter
	movwf	gpcnt2
ld200	movlw	0xC8		;load inner counter
	movwf	gpcnt		;= 200 dec
dly200	call	dly1ms
	decfsz	gpcnt,f		;done 200 ?
	goto	dly200		;no
	decfsz	gpcnt2,f	;yes, done 600 ?
	goto	ld200		;no, do another 200
	return

;***************SUB TO PROVIDE A 1mSec DELAY**************************************
;delay loop designed to provide a 1mSec delay including the call to this sub,
;setting up the counter, and the "return" to the caller 
dly1ms	movlw	0xF8		;= 248
	movwf	dcnt
	nop			;nops added to pad out delay time
	nop
dloop	nop			;this loop is 4 cycles (4 uSec) long
	decfsz	dcnt,f
	goto	dloop	
	return
	
;***************INTERRUPT SERVICE ROUTINE*****************************************
;One of two events can get us here:
;A rising edge on the external interrupt input (used here as serial input line)
;or a time out whilst waiting for the next bit in a byte (loss of byte sync).
;NOTE "SWAPF" is used so that the state of the "Z" flag is preserved
isr	movwf   wtemp       	;save current W register contents
	swapf	STATUS,w  	;swap status register into W 
	bcf	STATUS,RP0 	;select bank 0
	movwf	statemp    	;save contents of STATUS register in bank 0
	bcf	T1CON,TMR1ON	;always stop timer (prevent false time out)
;NOTE because timer registers only INCREMENT we have to set the values to 
;full count - the required count ie 65536 - 2500 = 63036 (0xF63C)
	movlw	0x3C		;load it for next time (2.5mSec) 
	movwf	TMR1L		;low byte
	movlw	0xF6
	movwf	TMR1H		;high byte
	btfsc	INTCON,INTF	;interrupt caused by rising edge on ext int?
	goto	getbit		;yes
;No rising edge has occurred within  the time out period so signal a framing error
	bsf	ferflg
	bcf	busyflg		;clr busy flag
	bcf	PIR1,TMR1IF	;clr timer flag
	goto	exisr	
;Rising edge, so start processing new bit
;if busy flag is not set then this is the start of a new byte so the byte shift
;register (inbyte) is set to 00000001, which on the the 8th shift will leave the 
;carry bit set to signal that the last bit of the current byte has been read	
getbit	btfsc	busyflg		;is busy flag set ?
	goto	rdnxbit		;yes
	clrf	inbyte		;no, init byte shift reg
	incf	inbyte,f
	bsf	busyflg		;set busy flag
rdnxbit	clrf	bitsac		;clr '1's count
	movlw	0xFA		; = 250 decimal
	movwf	samcnt		;init sample count
sertst	btfsc	serin		;serial input line = '1' ?
	incf	bitsac,f	;yes, incr '1's count
	nop			;use NOPs to make delay to next sample
	nop			;10uSec.
	nop			
	nop
	nop
	decfsz	samcnt,f	;done 250 samples ?
	goto	sertst		; no, sample input line again
	movlw	0x5A		;= 90 decimal
	subwf	bitsac,w	;test 1's value
	btfsc	STATUS,C	;val < 90 ?
	goto	bitest		;no, test for '0' or '1'
	bsf	berflg		;yes, set bit error flag
	bcf	busyflg		;clr busy flag
	goto	exisr		;error was probably interference so exit	
bitest	movlw	0xBE		;= 190 decimal
	subwf	bitsac,w	;bit = '0' or '1' (result is in carry)
bitrot	rlf	inbyte,f	;rotate bit into byte shift reg  LSB
;NOTE if the carry is now set, this was the last bit
	btfss	STATUS,C	;last bit ?
	goto	stime		;no
	bcf	busyflg		;yes, clr busy flag
	bsf	brdyflg		;set byte ready flag
	goto	exisr
stime	bsf	T1CON,TMR1ON	;start timer	
exisr	bcf	INTCON,INTF	;clr external interrupt flag, exit isr
	swapf   statemp,w 	;retrieve copy of STATUS register
	movwf	STATUS		;restore pre-isr STATUS register contents
	swapf   wtemp,f
	swapf   wtemp,w  	;restore pre-isr W register contents
	retfie 			;return from interrupt
;*********************************************************************************
	END