;*********************************************************************************
;          
;This program is part of a system to obtain Video images from a Video camera
;mounted in a bird box via a 2.4Ghz wireless channel.
;
;The part of the system at the camera end (called the TX) is controlled by
;commands sent at the other end (called the RX) via a 433Mhz radio channel.
;If the command is received and understood correctly, the TX will then respond by
;repeating the command back to the RX via the audio channel of the TX, this
;provides a crude "handshake" system so that the RX does not send any more
;commands until the TX has responded.
;Since a desktop or laptop PC is to be used to  monitor the video (via a video
;to USB converter) and display the battery  voltage and to issue the commands
;to the RX system (for onwards transmission to TX system) the RX system communicates
;with the PC using the USART of the RX PIC (RS232)
;
;At present, the only commands are to turn the TX system on and off and to
;request the TX system to send battery voltage data to the RX. 
;
;NOTE.  The battery voltage is only present at the A/D input when camera and video
;TX are swiched on and since the TX can only reply to commands via the TX audio 
;channel it can only reply if it is already ON or the command is to turn it ON.
;
;This is the code for the RX system which uses the PIC16F687. Unless there are
;communication errors  or a coms test byte (HEX AA) is sent by the PC it mostly
;passes messages between the TX and PC  
; 
;As an afterthought Manual ON and OFF buttons have been added so that the systyem
;can be used without a PC to watch the video on a TV etc.
;                                                                   
;*********************************************************************************
;                                                                     
;    Filename:	    BcamRX.asm                                        
;    Date:          10/05/09                                                  
;    File Version:                                                    
;                                                                     
;    Author:                                                          
;    Company: 
;                                                       
;   Files Required: P16F87.INC   
;                                                                                                                                      
;*********************************************************************************
;                                                                                                           
;                                                                           
	list		p=16f87	; list directive to define processor
	#include	<P16F87.inc>	; processor specific variable definitions

	errorlevel  -302    	;suppress message 302 from list file
	

     __CONFIG    _CONFIG1, _CP_OFF & _CCP1_RB0 & _DEBUG_OFF & _WRT_PROTECT_OFF & _CPD_OFF & _LVP_OFF & _BODEN_OFF & _MCLR_ON & _PWRTE_ON & _WDT_OFF & _INTRC_IO
     __CONFIG    _CONFIG2, _IESO_OFF & _FCMEN_OFF


; '__CONFIG' directive is used to embed configuration data within .asm file.
; The labels following the directive are located in the respective .inc file.
; See respective data sheet for additional information on configuration word.
;
;*********************RESOURCES USED**********************************************
;Internal 4MHz oscillator
;
;I/O PORTS:
;comparator C2 used to condition signal from CAM TX: NOTE, that at the comparator
;output, the signal sense is inverted, ie "1" = low level and "0" = high level.
;RA1 = comparator - input
;RA2 =     "      +   "
;RA4 =     "        output
;
;RA5 = MCLR
;
;RB0 = Manual camera ON button
;RB1 = Manual camera OFF button
;
;RB2 = USART RX
;RB5 = USART TX
;
;RB7 = serial output to CAM TX
;
;Ram used for variable storage
;
;Timer 1 used to provide  communication time out protection. 
;                                                                                                                         
; 
;********* VARIABLE STORAGE (RAM LOCATIONS)***************************************
wtemp   EQU     0x20 	      	;variable used for context saving W 
statemp EQU     0x21 	       	;variable used for context saving PIC STATUS 
inbyte	EQU	0x22		;input byte from TX
bitsac	EQU	0x23		;holds bit '1's count
samcnt	EQU	0x24		;holds the bit sample count
ipsflg	EQU	0x25		;input status flags
outbyte	EQU	0x26		;output byte to TX
battv	EQU	0x27		;battery voltage read from TX
dcnt	EQU	0x28		;counter used in time delay loop
dbcnt	EQU	0x29		;button de-bounce delay counter
gpcnt	EQU	0x2A		;general purpose counter
gpcnt2	EQU	0X2B		;Another counter
pcinb	EQU	0x2C		;store for byte from PC
intsave	EQU	0x2D		;to save "INTCON" state during byte transmission
;
;***********************DEFINITIONS***********************************************
#define brdyflg	ipsflg,0	;1 = input byte ready flag
#define busyflg	ipsflg,1	;1 = input is busy (receiving a byte)
#define berflg	ipsflg,2	;1 = input bit error (probably interference)
#define	ferflg	ipsflg,3	;1 = input framing error (lost byte sync)
;
;
#define manon	PORTB,0		;manual control on button
#define manoff	PORTB,1		;manual control off button
#define	serout	PORTB,7		;used for serial output (TO TX) 
;
;*********************************************************************************
	ORG     0x000		;processor reset vector
	goto    init 		;go to beginning of program
		
	ORG     0x004	 	;interrupt vector location
	Goto	isr
;*****************initialize I/O etc**********************************************
;The "_CONFIG" line above will have selected the internal oscillator block as the
;clock source.  We now set set CPU clock freqency to 4MHz so that it is the same
;as that of the BCAM TX and we can use some of the same timing routines.  After 
;this we init IO ports etc.
init	bsf	STATUS,RP0	;select reg bank 1
	movlw	0x60
	movwf	OSCCON		;set Frequency to 4 MHz
fwait	btfss	OSCCON,2	;frequency stable (IOFS bit set) yet ?
	goto	fwait		;no

;init ports
	bcf     STATUS,RP0	;yes, select reg bank  0
	movlw	0x10		;init port A latches
	movwf	PORTA		;only bit 4 high
	movlw	0x03		;init port B latches
	movwf	PORTB		;bits 0 and 1 high	
	bsf	STATUS,RP0	;sel bank1
	movlw	0x26		;init port A directions
	movwf	TRISA		;bits 1,2 and 5  = Input, rest output
	movlw	0x27		;init port B directions
	movwf	TRISB		;configure for USART and man cont button functions 
	bcf	OPTION_REG,7	;enable weak pull-ups for man cont buttons

;init USART:
;set baud rate to 9600
	movlw	0x19		;=25 decimal
	movwf	SPBRG
;enable USART TX
	movlw	0x24
	movwf	TXSTA		;TXEN and BRG =1
	bcf     STATUS,RP0	;set file register bank to 0
	bsf	PIR1,TXIF	;set USART TX ready flag
;enable USART RX
	movlw	0x90
	movwf	RCSTA		;SPEN and CREN  1

;configure comparator
	bsf	STATUS,RP0	;sel bank1
	movlw	6
	movwf	CMCON		;2 comparators to get output/s (C2)
	bcf     STATUS,RP0	;set file register bank to 0

; Clear all system variables
	movlw	intsave-wtemp	;= num of locations to be cleared
	movwf	wtemp
	movlw	statemp		;setup pointer
	movwf	FSR
clvars	clrf	INDF		;clear var
	incf	FSR,f		;increment pointer
	decfsz	wtemp,f		;cleared all vars ?
	goto	clvars		;no

;init interrupt sys: ONLY  on comparator output change and timer 1 overflow
	clrf	PIR1		;clear flags
	clrf	PIR2
	bsf	STATUS,RP0	;sel bank1	
	movlw	0x01
	movwf	PIE1		;enable timer1 int
	movlw	0x40
	movwf	PIE2		;enable comparator int
	bcf     STATUS,RP0	;set file register bank to 0

;Untill the TX is turned on, a considerable amount of noise is present on the
;audio channel, which would cause many false interrupts; for this reason,
;interrupts are globaly dissabled at this end untill 0.5sec after any command to
;turn the the TX on has been sent.
;Naturaly interrupts will be turned off again when a byte signifying the the TX is
;about to be turned off (in response to a command or because of a low battery)
;has been received.

;***********************THE EXECUTIVE OR RUNNING PART*****************************
;see if there is any input from PC or manual buttons and  check for any comms 
;error/s
run	btfsc	manon		;manual ON button pressed (input LOW)?
	goto	mofftst		;no
	call	delay25		;yes, wait 25 mSec
	btfsc	manon		;button input still low ?
	goto	mofftst		;no
	movlw	0xC1		;yes get camera on command byte
	movwf	pcinb		;pretend it came from PC
	goto	sbctx		;send it to TX
mofftst	btfsc	manoff		;manual off button pressed ?
	goto	pcitst		;no
	call	delay25
	btfsc	manoff
	goto	pcitst
	movlw	0xC0		;get camera off command byte
	movwf	pcinb
	goto	sbctx		;send to TX		
pcitst	btfss	PIR1,5		;any input from PC ?
	goto	bcintst		;no
	btfss	RCSTA,2		;yes, framing error ?
	goto	oertst		;no
	movf	RCREG,w		;yes read rcreg to reset error flag
	movlw	0xE0		;tell PC
	call	pcsend	
	goto	run
oertst	btfss	RCSTA,1		;overrun error ?
	goto	readpc		;no, read byte from pc
	bcf	RCSTA,4		;yes, clear then set bit "CREN" to clear error flg
	bsf	RCSTA,4	
	movlw	0xE1		;tell pc
pcrep	call	pcsend
	goto	run
;Read input byte from PC
readpc	movf	RCREG,w		;read byte
	movwf	pcinb		;save byte
	xorlw	0xAA
	btfss	STATUS,Z	;commms test byte ?
	goto	sbctx		;no, send byte to BCAMTX
	movf	pcinb,w		;yes reply to PC
	goto	pcrep		
sbctx	movf	pcinb,w		;send byte to BCAMTX	
	call	sbyte
;If byte was command to turn TX on, wait for 0.5 seconds then enable interrupts
	movf	pcinb,w	
	xorlw	0xC1
	btfss	STATUS,Z	;command to turn TX on ?
	goto	run		;no
	movlw	02		;yes load outer counter
	movwf	gpcnt2
ld250	movlw	0xFA		;load inner counter
	movwf	gpcnt		;= 250 dec
dly250	call	dly1ms
	decfsz	gpcnt,f		;done 250 ?
	goto	dly250		;no
	decfsz	gpcnt2,f	;yes, done 500 ?
	goto	ld250		;no, do another 250
	clrf	PIR1		;yes clear flags, then enable interrupts
	clrf	PIR2
	movlw	0xC0
	movwf	INTCON		;set GIE,PEIE (for timer1) and comp output change	
	goto	run		
;Check for reply or error message input from TX
;Note errors E0 to E7 are generated at this end, E8 to EF are generated
;at the TX end
bcintst	movf	ipsflg,f	;move file to self to test for 0
	btfsc	STATUS,Z	;any input status flags set ?
	goto	run		;no
;TX input flag/s set find which one/s
bflgtst	btfsc	busyflg		;busy flag ?
	goto	bflgtst		;yes
	btfss	brdyflg		;no, byte ready flag 	
	goto	rxerr		;no, must be error
	movf	inbyte,w	;yes send byte to PC
	call	pcsend
	clrf	ipsflg		;clr all input flags
txotst	xorlw	0xC0
	btfss	STATUS,Z	;command = turn TX off ?
	goto	txbtst		;no
intoff	clrf	INTCON		;dissable interrupts
	goto	run
txbtst	xorlw	0xE8
	btfss	STATUS,Z	;= battery low error ?
	goto	run		;no
	goto	intoff		;yes, dissable interrupts
rxerr	movlw	0xE2
	btfsc	berflg		;bit error ?
	goto	senderr		;yes
	movlw	0xE3		;no, must be framing error
senderr	clrf	ipsflg		;clear all input flags
	call	pcsend		;tell pc about error
	goto	run		

;*********SUB ROUTINE TO SEND A BYTE TO BCAM TX **********************************
;enter with byte in W reg
;we dissable interrupts when sending bytes so that interrupts do not interfere
;with bit timing.
sbyte	movwf	outbyte		;save byte
	movf	INTCON,w	;save interrupt enable state
	movwf	intsave
	clrf	INTCON		;dissable ints	
	movlw	8
	movwf	gpcnt		;to count 8 bits
sbnxb	rlf	outbyte,f	;rotate bit into carry
	bsf	serout		;bring serial out line high
	call	dly1ms		;delay 1 mSec
	btfss	STATUS,C	;bit = 1 ?
	bcf	serout		;no, bring serial out line low
	call	dly1ms		;delay 1ms
	bcf	serout		;allways bring serial out line low at this time
	call	dly1ms		;1 mS to next bit or end of byte
	decfsz	gpcnt,f		;done 8 bits ?
	goto	sbnxb		;no
	bcf	PIR1,TMR1IF	;yes, clear timer flag and
	bcf	PIR2,CMIF	;clear comparator flag	
	movf	intsave,w	;now restore int enable state 
	movwf	INTCON	
	return

;***************SUB TO PROVIDE A 1mSec DELAY**************************************
;delay loop designed to provide a 1mSec delay including the call to this sub,
;setting up the counter, and the "return" to the caller 
dly1ms	movlw	0xF8		;= 248
	movwf	dcnt
	nop			;nops added to pad out delay time
	nop
dloop	nop			;this loop is 4 cycles (4 uSec) long
	decfsz	dcnt,f
	goto	dloop	
	return

;**********Sub to provide a 25mSec delay to de-bounce man cont buttons************
delay25 movlw	0x19		;= 25 decimal
	movwf	dbcnt
d25loop	call	dly1ms
	decfsz	dbcnt,f		;done 25 1mSec loops ?
	goto	d25loop		;no
	return			;yes

;***************SUB T0 SEND A BYTE TO PC *****************************************
pcsend	btfss	PIR1,TXIF	;USART TX ready ?
	goto	pcsend		;no
	movwf	TXREG		;yes, send byte
	return
	
;***************INTERRUPT SERVICE ROUTINE*****************************************
;One of two events can get us here:
;A change in the state of comparator 2 output(used here as serial input from TX)
;or a time out whilst waiting for the next bit in a byte (loss of byte sync).
;NOTE "SWAPF" is used so that the state of the "Z" flag is preserved
isr	movwf   wtemp       	;save current W register contents
	swapf	STATUS,w  	;swap status register into W 
	bcf	STATUS,RP0 	;select bank 0
	movwf	statemp    	;save contents of STATUS register in bank 0
	bcf	T1CON,TMR1ON	;always stop timer (prevent false time out)
;NOTE because timer registers only INCREMENT we have to set the values to 
;full count - the required count ie 65536 - 2500 = 63036 (0xF63C)
	movlw	0x3C		;load it for next time (2.5mSec) 
	movwf	TMR1L		;low byte
	movlw	0xF6
	movwf	TMR1H		;high byte
	btfsc	PIR2,CMIF	;interrupt caused by comp output change?
	goto	getedge		;yes
;No ege has occurred within  the time out period so signal a framing error
	bsf	ferflg
	bcf	busyflg		;clr busy flag
	bcf	PIR1,TMR1IF	;clr timer flag
	goto	exisr	
;Note any comparator change (high to low or vice versa) will set flag so we have
;to first see if this is a falling edge
getedge	bsf	STATUS,RP0	;sel bank 1
	movf	CMCON,w		;rd cmcon to get comp state and remove mismatch
	bcf	STATUS,RP0	;sel bank 0
	bcf	PIR2,CMIF	;clear flag
	andlw	0x80
	btfss	STATUS,Z	;falling edge ?	
;Note comparator inverts the signal so low = logic "1" and falling edge = start
;of bit so start processing new bit and since the whole bit is processed within
;the isr we should not arrive here on a rising edge
	goto	biterr		;no	
;if busy flag is not set then this is the start of a new byte so the byte shift
;register (inbyte) is set to 00000001, which on the the 8th shift will leave the 
;carry bit set to signal that the last bit of the current byte has been read	
	btfsc	busyflg		;is busy flag set ?
	goto	rdnxbit		;yes
	clrf	inbyte		;no, init byte shift reg
	incf	inbyte,f
	bsf	busyflg		;set busy flag
rdnxbit	clrf	bitsac		;clr '1's count
	movlw	0xFA		; = 250 decimal
	movwf	samcnt		;init sample count
sertst	bsf	STATUS,RP0	;sel bank 1
	movf	CMCON,w		;rd cmcon to get comp state
	bcf	STATUS,RP0
	nop			;NOP used to pad loop timing (to 10uSec|)
	andlw	0x80
	btfsc	STATUS,Z	;serial input line = '1' ?
	incf	bitsac,f	;yes, incr '1's count
	decfsz	samcnt,f	;done 250 samples ?
	goto	sertst		; no, sample input line again
	movlw	0x5A		;= 90 decimal
	subwf	bitsac,w	;test 1's value
	btfsc	STATUS,C	;val < 90 ?
	goto	bitest		;no, test for '0' or '1'
biterr	bsf	berflg		;yes, set bit error flag
	bcf	busyflg		;clr busy flag
	goto	exisr		;error was probably interference so exit	
bitest	movlw	0xBE		;= 190 decimal
	subwf	bitsac,w	;bit = '0' or '1' (result is in carry)
bitrot	rlf	inbyte,f	;rotate bit into byte shift reg  LSB
;NOTE if the carry is now set, this was the last bit
	btfss	STATUS,C	;last bit ?
	goto	stime		;no
	bcf	busyflg		;yes, clr busy flag
	bsf	brdyflg		;set byte ready flag
	goto	exisr
stime	bsf	T1CON,TMR1ON	;start timer	
exisr	bcf	PIR2,CMIF	;clr comparator interrupt flag, exit isr
	swapf   statemp,w 	;retrieve copy of STATUS register
	movwf	STATUS		;restore pre-isr STATUS register contents
	swapf   wtemp,f
	swapf   wtemp,w  	;restore pre-isr W register contents
	retfie 			;return from interrupt
;*********************************************************************************
	END




