// This code has been ported from the tutorial code of James Lynch, and extended with bitmap display and
// "Very Large" font support by Mike Hibbett

#include "lcd_font.h"
#include "lcd_epson.h"


void SPISendByte( boolean mode, unsigned int data )
{
  unsigned int lp;
  
  digitalWrite(MOSI_PIN, mode);  // assert the D/C bit
  digitalWrite(SCLK_PIN, HIGH);  // toggle out the first bit
  digitalWrite(SCLK_PIN, LOW);
  
  for (lp=0; lp < 8; lp++) {
    // assert the most significant bit of the data byte
    if ( data & 0x80 ) {
      digitalWrite(MOSI_PIN, HIGH );  
    } else {
      digitalWrite(MOSI_PIN, LOW );  
    }
    
    // move the bits in the data byte up one bit.
    data = data << 1;
    
    digitalWrite(SCLK_PIN, HIGH);  // toggle out the first bit
    digitalWrite(SCLK_PIN, LOW);
  }
}


void LCDSetup( void )
{
  // Set SPI bus initial level:
  //    backlight high ( on, it needs to be to see anything )
  //    CS high ( LCD disabled )
  //    RESET low ( LCD held in reset )
  //    SCLK low ( normal default state )
  //    MOSI low ( doesn't matter, but set it to something! )
  pinMode(BACKLIGH_PIN, OUTPUT);     
  pinMode(CS_PIN, OUTPUT);
  pinMode(RESET_PIN, OUTPUT);
  pinMode(SCLK_PIN, OUTPUT);
  pinMode(MOSI_PIN, OUTPUT);
  
  digitalWrite(CS_PIN, HIGH);
  digitalWrite(RESET_PIN, LOW);
  digitalWrite(SCLK_PIN, LOW);
  digitalWrite(MOSI_PIN, LOW);
  digitalWrite(BACKLIGH_PIN, HIGH);

  // Reset the LCD
  delay(5);  // Wait for a short period to ensure the LCD is correctly reset
  digitalWrite(RESET_PIN, HIGH);
  
  // Enable LCD
  digitalWrite(CS_PIN, LOW);
    
  // Turn the display on
  SPISendByte(COMMAND_MODE, 0xCA);  
  SPISendByte(DATA_MODE, 0x00);  
  SPISendByte(DATA_MODE, 0x20);  
  SPISendByte(DATA_MODE, 0x00);  
  
  SPISendByte(COMMAND_MODE, 0xBB);  
  SPISendByte(DATA_MODE, 0x01);  
  
  SPISendByte(COMMAND_MODE, 0xD1);  

  SPISendByte(COMMAND_MODE, 0x94);  

  SPISendByte(COMMAND_MODE, 0x20);  
  SPISendByte(DATA_MODE, 0x0f);  
  
  // Inverse display colours
  SPISendByte(COMMAND_MODE, DISINV);

  SPISendByte(COMMAND_MODE, 0xBC);  
  SPISendByte(DATA_MODE, 0x01);  
  SPISendByte(DATA_MODE, 0x00);  
  SPISendByte(DATA_MODE, 0x02);  
  
  SPISendByte(COMMAND_MODE, 0x81);  
  SPISendByte(DATA_MODE, 0x6B);  // contrast
  SPISendByte(DATA_MODE, 0x03);  
    
  LCDCLS(BLACK);
  
  // finally, display on
  SPISendByte(COMMAND_MODE, 0xAF);    
}


void LCDCLS( unsigned int color )
{
  unsigned int lp;
  
  SPISendByte(COMMAND_MODE, 0x75);
  SPISendByte(DATA_MODE, 0);
  SPISendByte(DATA_MODE, 131);
  SPISendByte(COMMAND_MODE, 0x15);
  SPISendByte(DATA_MODE, 0);
  SPISendByte(DATA_MODE, 131);
  
  SPISendByte(COMMAND_MODE, 0x5C);
  
  for (lp=0; lp < (( 131 * 131 ) / 2 ); lp++ ) {
    SPISendByte(DATA_MODE, (color>>4) & 0xFF);    
    SPISendByte(DATA_MODE, ((color & 0x0F) << 4) | ((color >> 8) & 0x0F));    
    SPISendByte(DATA_MODE, color & 0xFF);    
  }  
}


void LCDPutBMP( unsigned char *bitmap, unsigned int x, unsigned int y, unsigned int iwidth, unsigned int idepth)
{
  unsigned int lp;
  unsigned char color1, color2;
  
  SPISendByte(COMMAND_MODE,PASET);
  SPISendByte(DATA_MODE,x);
  SPISendByte(DATA_MODE,x+idepth-1);
  // Column address set (command 0x2A)
  SPISendByte(COMMAND_MODE,CASET);
  SPISendByte(DATA_MODE,y);
  SPISendByte(DATA_MODE,y+iwidth-1);
  
  SPISendByte(COMMAND_MODE,RAMWR);

  for ( lp=0; lp < ((iwidth * idepth)/2); lp++ ) {
    color1 = *bitmap++;
    color2 = *bitmap++;
    SPISendByte(DATA_MODE,(color1 >> 4) & 0xFF);
    SPISendByte(DATA_MODE,((color1 & 0xF) << 4) | ((color2 >> 8) & 0xF));
    SPISendByte(DATA_MODE,color2 & 0xFF);
  }
}


void LCDSetPixel(unsigned int x, unsigned int y, unsigned int color) 
{
  SPISendByte(COMMAND_MODE,PASET);
  SPISendByte(DATA_MODE,x);
  SPISendByte(DATA_MODE,x);
  // Column address set (command 0x2A)
  SPISendByte(COMMAND_MODE,CASET);
  SPISendByte(DATA_MODE,y);
  SPISendByte(DATA_MODE,y);
  // Now illuminate the pixel (2nd pixel will be ignored)
  SPISendByte(COMMAND_MODE,RAMWR);
  SPISendByte(DATA_MODE,(color >> 4) & 0xFF);
  SPISendByte(DATA_MODE,((color & 0xF) << 4) | ((color >> 8) & 0xF));
  SPISendByte(DATA_MODE,color & 0xFF);
}


void LCDSetLine(unsigned int x0, unsigned int y0, unsigned int x1, unsigned int y1, unsigned int color) 
{
  int dy = y1 - y0;
  int dx = x1 - x0;
  int stepx, stepy;
  
  if (dy < 0) { 
    dy = -dy; 
    stepy = -1; 
  } else { 
    stepy = 1; 
  }
  
  if (dx < 0) { 
    dx = -dx; 
    stepx = -1; 
  } else { 
    stepx = 1; 
  }
  
  dy <<= 1; // dy is now 2*dy  
  dx <<= 1; // dx is now 2*dx

  LCDSetPixel(x0, y0, color);

  if (dx > dy) {
    int fraction = dy - (dx >> 1); // same as 2*dy - dx

    while (x0 != x1) {
      if (fraction >= 0) {
        y0 += stepy;
        fraction -= dx; // same as fraction -= 2*dx
      }

      x0 += stepx;
      fraction += dy; // same as fraction -= 2*dy
      LCDSetPixel(x0, y0, color);
    }
  } else {
    int fraction = dx - (dy >> 1);

    while (y0 != y1) {
      if (fraction >= 0) {
        x0 += stepx;
        fraction -= dy;
      }

      y0 += stepy;
      fraction += dx;
      LCDSetPixel(x0, y0, color);
    }
  }
}


void LCDPutChar(char c, int x, int y, int size, int fColor, int bColor) 
{
  int i,j;
  int mult;
  unsigned int nCols;
  unsigned int nRows;
  unsigned int nBytes;
  unsigned char PixelRow;
  unsigned char Mask;
  unsigned int Word0;
  unsigned int Word1;
  unsigned char *pFont;
  unsigned char *pChar;
  unsigned char *FontTable[] = {(unsigned char *)FONT6x8, (unsigned char *)FONT8x8, (unsigned char *)FONT8x16};
  
  // HUG font is simply the LARGE font doubled up
  if ( size == VLARGE ) {
    size = LARGE;
    mult = 2;  
  } else {
    mult = 1;
  }
  
  // get pointer to the beginning of the selected font table
  pFont = (unsigned char *)FontTable[size];

  // get the nColumns, nRows and nBytes
  nCols = *pFont;
  nRows = *(pFont + 1);
  nBytes = *(pFont + 2);

  // get pointer to the last byte of the desired character
  pChar = pFont + (nBytes * (c - 0x1F)) + nBytes - 1;

  // Row address set (command 0x2B)
  SPISendByte(COMMAND_MODE,PASET);
  SPISendByte(DATA_MODE,x);
  SPISendByte(DATA_MODE,x + (nRows*mult) - 1);

  // Column address set (command 0x2A)
  SPISendByte(COMMAND_MODE,CASET);
  SPISendByte(DATA_MODE,y);
  SPISendByte(DATA_MODE,y + (nCols*mult) - 1);

  // WRITE MEMORY
  SPISendByte(COMMAND_MODE,RAMWR);

  // loop on each row, working backwards from the bottom to the top
  for (i = nRows - 1; i >= 0; i--) {
    // copy pixel row from font table and then decrement row
    PixelRow = *pChar--;
   
    if ( mult == 1 ) {
      // loop on each pixel in the row (left to right)
      // Note: we do two pixels each loop
      Mask = 0x80;
      
      for (j = 0; j < nCols; j += 2) {
        // if pixel bit set, use foreground color; else use the background color
        // now get the pixel color for two successive pixels
        if ((PixelRow & Mask) == 0)
          Word0 = bColor;
        else
          Word0 = fColor;
  
        Mask = Mask >> 1;
  
        if ((PixelRow & Mask) == 0)
          Word1 = bColor;
        else
          Word1 = fColor;
  
        Mask = Mask >> 1;
  
        // use this information to output three data bytes
        SPISendByte(DATA_MODE,(Word0 >> 4) & 0xFF);
        SPISendByte(DATA_MODE,((Word0 & 0xF) << 4) | ((Word1 >> 8) & 0xF));
        SPISendByte(DATA_MODE,Word1 & 0xFF);
      }
    } else {
      // loop on each pixel in the row (left to right)
      // Note: we do one pixel each loop, printing it twice
      Mask = 0x80;

      for (j = 0; j < nCols; j++) {
        // if pixel bit set, use foreground color; else use the background color
        // now get the pixel color for two successive pixels
        if ((PixelRow & Mask) == 0)
          Word0 = bColor;
        else
          Word0 = fColor;
  
        Mask = Mask >> 1;
  
        // use this information to output three data bytes
        SPISendByte(DATA_MODE,(Word0 >> 4) & 0xFF);
        SPISendByte(DATA_MODE,((Word0 & 0xF) << 4) | ((Word0 >> 8) & 0xF));
        SPISendByte(DATA_MODE,Word0 & 0xFF);
      }
      Mask = 0x80;

      for (j = 0; j < nCols; j++) {
        // if pixel bit set, use foreground color; else use the background color
        // now get the pixel color for two successive pixels
        if ((PixelRow & Mask) == 0)
          Word0 = bColor;
        else
          Word0 = fColor;
  
        Mask = Mask >> 1;
  
        // use this information to output three data bytes
        SPISendByte(DATA_MODE,(Word0 >> 4) & 0xFF);
        SPISendByte(DATA_MODE,((Word0 & 0xF) << 4) | ((Word0 >> 8) & 0xF));
        SPISendByte(DATA_MODE,Word0 & 0xFF);
      }
    }
  }

  // terminate the Write Memory command
  SPISendByte(COMMAND_MODE,NOP);
}


void LCDPutStr(char *pString, unsigned int x, unsigned int y, unsigned int Size, unsigned int fColor, unsigned int bColor) 
{
  // loop until null-terminator is seen
  while (*pString != 0x00) {
    // draw the character
    LCDPutChar(*pString++, x, y, Size, fColor, bColor);
    
    // advance the y position
    if (Size == SMALL)
      y = y + 6;
    else if (Size == MEDIUM)
      y = y + 8;
    else if (Size == LARGE)
      y = y + 8;
    else 
      y = y + 16;

    // bail out if y exceeds 131
    if (y > 131) 
      break;
  }
}

