    ;*************************************************************************
	; file: LCD.asm
	;		Contains the code to handle an attached LCD module based on the
	;		44780 driver, used in 4-bit mode.
	;*************************************************************************


    ;*************************************************************************
	; include files go here
	;*************************************************************************

    include <P18f27J13.INC>
	include <macros.inc>


	;*************************************************************************
	; extern directives - to gain access to global functions and data from 
	;					  elsewhere
	;*************************************************************************

	extern usleep


	;*************************************************************************
	; global directives - making variables and functions available elsewhere -
	;					  go here
	;*************************************************************************

	global	LCDInit
	global	LCDCol
	global	LCDLine
	global	setPos
	global	LCDWrite


	;*************************************************************************
	; Constanst - Symbolic constants and simple macros, defined with the 
	;			  EQU directive, go here.
	;*************************************************************************

	; LCD pin control
	LCD_RS_LAT		EQU LATA
	LCD_RS_BIT		EQU 0
	LCD_RW_LAT		EQU LATA
	LCD_RW_BIT		EQU 1
	LCD_E_LAT		EQU LATA
	LCD_E_BIT		EQU 2
	LCD_DATA_LAT	EQU LATB	; We use the low 4 bits.
	
	; LCD position constants
	LCD_LINE1		EQU	0
	LCD_LINE2		EQU 0x40


	;*************************************************************************
	; Definition of RAM based variables follow the udata directive
	;*************************************************************************

	udata

wnTmp			res	1	; Temp variable used by writeNibble
LCDLine			res	1	; Input to setPos routine
LCDCol			res	1	; Input to setPos
wlTmp			res	1	; Temp variable for LCDWrite


	;*************************************************************************
	; code, such as functions and ROM based data tables, follow the code directive
	;*************************************************************************

	code
	
	;*************************************************************************
	; Function: writeNibble.
	;			Writes a nibble to the LCD ( we use the LCD in 4-bit mode )
	; 			We make use of the fact that the LATB lower four bits are 
	;			always 0 on entry.
	;*************************************************************************
writeNibble:	
	movwf	wnTmp

	btfsc	wnTmp, 4	; bit 4 is the state of the RW line
	bsf		LCD_RW_LAT, LCD_RW_BIT
	btfsc	wnTmp, 5	; bit 5 is the state of the RS line
	bsf		LCD_RS_LAT, LCD_RS_BIT
	movlw	0x0F
	andwf	wnTmp, W	; remove the two control bits from the upper nibble..

	; now copy the data over to the lower four bits of PORTB, preserving the
	; upper parts of PORTB
	iorwf	LCD_DATA_LAT, F
	
	USLEEP	USLEEP_2US
	bsf		LCD_E_LAT, LCD_E_BIT 
	USLEEP	USLEEP_2US
	bcf		LCD_E_LAT, LCD_E_BIT 
	
	; drop all the LCD lines to low level
	bcf		LCD_RW_LAT, LCD_RW_BIT
	bcf		LCD_RS_LAT, LCD_RS_BIT
	movf	LCD_DATA_LAT, W
	andlw	0xF0
	movwf	LCD_DATA_LAT
	
	return 	
	
	
	;*************************************************************************
	; Function:	setPos
	;			Sets the position on the display.
	; 			Expects the variables LCDLine and LCDCol to have been set.
	; 			These take the values 0..15 and LCD_LINE1,LCD_LINE2 respectively
	;*************************************************************************
setPos:
	movf 	LCDLine, W
	addwf	LCDCol, F		; Destroys the original LCDCol, but thats ok
	swapf	LCDCol, W
	iorlw	0x08
	andlw	0x0F
	call	writeNibble
	movf	LCDCol, W
	andlw	0x0F
	call	writeNibble
	USLEEP	USLEEP_100US
	
	return	
	

	;*************************************************************************
	; Function: LCDWrite
	;			Writes a character in W to the LCD at the current position
	;*************************************************************************
LCDWrite:
	movwf	wlTmp
	swapf	wlTmp, W
	andlw	0x0F
	iorlw	0x20
	call	writeNibble
	movf	wlTmp, W
	andlw	0x0F
	iorlw	0x20
	call	writeNibble

	USLEEP	USLEEP_100US
		
	return	
	
	
	;*************************************************************************
	; Function: LCDInit
	;			Setup the LCD to its default state following power up
	;*************************************************************************
LCDInit:
	movlw   0x00	
	movwf   TRISB		; Set PORTB as an output
	movlw   0xF8	
	movwf   TRISA		; Set lower 3 bits of PORTA as an output

	; Set our LCD drive pins to digital I/O pins
  	movlw  	0x07
   	movwf  	ANCON0
   	movlw  	0x17
   	movwf  	ANCON1

   	; Set all LCD signals low, and turn LED off
   	clrf	LATB
   	clrf	LATA

	; allow LCD time to start up
	USLEEP	USLEEP_80MS
	
	movlw	0x03
   	call	writeNibble

	USLEEP	USLEEP_8MS
	
	movlw	0x03
   	call	writeNibble

	USLEEP	USLEEP_200US
	
	movlw	0x03
   	call	writeNibble

	USLEEP	USLEEP_200US
	
	movlw	0x02
   	call	writeNibble

	USLEEP	USLEEP_200US
	
	movlw	0x02
   	call	writeNibble
	movlw	0x0c
   	call	writeNibble

	USLEEP	USLEEP_3MS
	
	movlw	0x00
   	call	writeNibble
	movlw	0x0c
   	call	writeNibble

	USLEEP	USLEEP_3MS
	
	movlw	0x00
   	call	writeNibble
	movlw	0x01
   	call	writeNibble

	USLEEP	USLEEP_3MS
	
	movlw	0x00
   	call	writeNibble
	movlw	0x06
   	call	writeNibble

	USLEEP	USLEEP_3MS
	
	return


	end
