    ;*************************************************************************
	; file: LCD.asm
	;		Contains the code to handle an attached LCD module based on the
	;		44780 driver, used in 4-bit mode.
	;*************************************************************************


    ;*************************************************************************
	; include files go here
	;*************************************************************************

    include <P18f27J13.INC>
	include <macros.inc>


	;*************************************************************************
	; extern directives - to gain access to global functions and data from 
	;					  elsewhere
	;*************************************************************************

	extern usleep


	;*************************************************************************
	; global directives - making variables and functions available elsewhere -
	;					  go here
	;*************************************************************************

	global	LCDInit
	global	LCDCol
	global	LCDLine
	global	setPos
	global	LCDWrite
	global 	bin2ASCIIh
	global 	LCD2ASCII
	global 	LCDOn
	global 	LCDOff
	global	LCDCursorOn
	global	LCDCursorOff


	;*************************************************************************
	; Constanst - Symbolic constants and simple macros, defined with the 
	;			  EQU directive, go here.
	;*************************************************************************

	; LCD pin control
LCD_RS_LAT		EQU LATB
LCD_RS_BIT		EQU 2
LCD_RW_LAT		EQU LATB
LCD_RW_BIT		EQU 1	
LCD_E_LAT		EQU LATB
LCD_E_BIT		EQU 0
LCD_DATA_LAT	EQU LATC	; We use the upper 4 bits.
	

	;*************************************************************************
	; Definition of RAM based variables follow the udata directive
	;*************************************************************************

	udata

wnTmp			res	1	; Temp variable used by writeNibble
LCDLine			res	1	; Input to setPos routine
LCDCol			res	1	; Input to setPos
wlTmp			res	1	; Temp variable for LCDWrite
ASCII_LOW		res	1	; Binary to Ascii conversion
ASCII_HIGH		res	1	; Binary to Ascii conversion: high character


	;*************************************************************************
	; code, such as functions and ROM based data tables, follow the code directive
	;*************************************************************************

	code
	


	;*************************************************************************
	; Function: bin2ASCIIh
	;			converts a binary value in W into two digit hex ASCII
	;			in ASC_HIGH and ASCI_LOW. Naturally, values between 0..99 only
	;			Therefore this is great for clock and timer usage ( if you are
	;			using BCD counting, or like your displays in hexidecimal )
	;*************************************************************************
bin2ASCIIh:
	movwf  	ASCII_HIGH
    swapf  	ASCII_HIGH, W
    andlw  	0x0f

    addlw  	6
    btfsc 	STATUS, DC
    addlw   'A' - ('9' + 1)
    addlw  	'0' - 6

    xorwf  	ASCII_HIGH, W
    xorwf  	ASCII_HIGH, F
    xorwf  	ASCII_HIGH, W

    andlw  	0x0f

    addlw  	6
    btfsc 	STATUS, DC
    addlw   'A' - ('9' + 1)
    addlw   '0' - 6
	movwf	ASCII_LOW

	return


	;*************************************************************************
	; Function: LCD2ASCII
	;			displays the two characters in ASCII_HIGH and ASCII_LOW.
	;			Typically, after converting a binary number using bin2ASCII
	;*************************************************************************
LCD2ASCII:
	movf	ASCII_HIGH, W
	call	LCDWrite
	movf	ASCII_LOW, W
	call	LCDWrite
	return


	;*************************************************************************
	; Function: writeNibble.
	;			Writes a nibble to the LCD ( we use the LCD in 4-bit mode )
	; 			We make use of the fact that the LATC upper four bits are 
	;			always 0 on entry.
	;*************************************************************************
writeNibble:	
	movwf	wnTmp

	btfsc	wnTmp, 4	; bit 4 is the state of the RW line
	bsf		LCD_RW_LAT, LCD_RW_BIT
	btfsc	wnTmp, 5	; bit 5 is the state of the RS line
	bsf		LCD_RS_LAT, LCD_RS_BIT
	movlw	0x0F
	andwf	wnTmp, F	; remove the two control bits from the upper nibble..
	swapf	wnTmp, W	; get the nibble into the upper nibble

	; now copy the data over to the upper four bits of PORTC, preserving the
	; lower parts of PORTC
	iorwf	LCD_DATA_LAT, F
	
	USLEEP	USLEEP_2US
	bsf		LCD_E_LAT, LCD_E_BIT 
	USLEEP	USLEEP_2US
	bcf		LCD_E_LAT, LCD_E_BIT 
	
	; drop all the LCD lines to low level
	bcf		LCD_RW_LAT, LCD_RW_BIT
	bcf		LCD_RS_LAT, LCD_RS_BIT
	movf	LCD_DATA_LAT, W
	andlw	0x0F
	movwf	LCD_DATA_LAT
	
	return 	
	
	
	;*************************************************************************
	; Function:	setPos
	;			Sets the position on the display.
	; 			Expects the variables LCDLine and LCDCol to have been set.
	; 			These take the values 0..15 and LCD_LINE1,LCD_LINE2 respectively
	;*************************************************************************
setPos:
	movf 	LCDLine, W
	addwf	LCDCol, F		; Destroys the original LCDCol, but thats ok
	swapf	LCDCol, W
	iorlw	0x08
	andlw	0x0F
	call	writeNibble
	movf	LCDCol, W
	andlw	0x0F
	call	writeNibble
	USLEEP	USLEEP_100US
	
	return	
	

	;*************************************************************************
	; Function: LCDWrite
	;			Writes a character in W to the LCD at the current position
	;*************************************************************************
LCDWrite:
	movwf	wlTmp
	swapf	wlTmp, W
	andlw	0x0F
	iorlw	0x20
	call	writeNibble
	movf	wlTmp, W
	andlw	0x0F
	iorlw	0x20
	call	writeNibble

	USLEEP	USLEEP_100US
		
	return	


	;*************************************************************************
	; Function: LCDCursorOn
	;			Turns on the cursor at the position specified in LCDCol,LCDLine
	;*************************************************************************
LCDCursorOn:
	call	setPos
	movlw	0x00
	call	writeNibble
	movlw	0x0D
	call	writeNibble
	USLEEP	USLEEP_100US
	return


	;*************************************************************************
	; Function: LCDCursorOff
	;			Turns the cursor off, leaves display on
	;*************************************************************************
LCDCursorOff:
	movlw	0x00
	call	writeNibble
	movlw	0x0C
	call	writeNibble
	USLEEP	USLEEP_100US
	return
	

	;*************************************************************************
	; Function: LCDOn LCDOff
	;			Turns the display on/off from deep low power down.
	;*************************************************************************
LCDOn:
	call	LCDInit

	return

LCDOff:
   	; Set all LCD signals low, and turn LCD off
   	clrf	LATB
   	clrf	LATC

	return

	
	;*************************************************************************
	; Function: LCDInit
	;			Setup the LCD to its default state following power up
	;*************************************************************************
LCDInit:
	movlw   0xF8	
	andwf   TRISB		; Set lower 3 bits of PORTB as an output
	movlw   0x0F	
	andwf   TRISC		; Set upper 4 bits of PORTC as an output

	bcf		TRISC, 2	; This bit controls the LCD power

   	; Set all LCD signals low, and turn LED off
   	clrf	LATB
   	clrf	LATC

	bsf		LATC, 2		; Power up the LCD

	; allow LCD time to start up
	USLEEP	USLEEP_80MS
	
	movlw	0x03
   	call	writeNibble

	USLEEP	USLEEP_8MS
	
	movlw	0x03
   	call	writeNibble

	USLEEP	USLEEP_200US
	
	movlw	0x03
   	call	writeNibble

	USLEEP	USLEEP_200US
	
	movlw	0x02
   	call	writeNibble

	USLEEP	USLEEP_200US
	
	movlw	0x02
   	call	writeNibble
	movlw	0x0c
   	call	writeNibble

	USLEEP	USLEEP_3MS
	
	movlw	0x00
   	call	writeNibble
	movlw	0x0c
   	call	writeNibble

	USLEEP	USLEEP_3MS
	
	movlw	0x00
   	call	writeNibble
	movlw	0x01
   	call	writeNibble

	USLEEP	USLEEP_3MS
	
	movlw	0x00
   	call	writeNibble
	movlw	0x06
   	call	writeNibble

	USLEEP	USLEEP_3MS
	
	return


	end
