;*****************************************************************************        
;
;   Module:     lplcd.asm
;               
;   Author:     Mike Hibbett 
;                                                                  
;   Version:    0.1 29/10/07                                                  
;
;               Demonstrates driving a low power, static LCD panel by
;               implementing a digital clock
;
;*****************************************************************************        

    list p=16F917    

	; Suppress the display of informational warning messages
    errorlevel -302
    errorlevel -306

    #include <p16f917.inc>
    


;*****************************************************************************        
;
;   Function :  Reset vector
;               Hardware entry point to the code
;               Just initialises the hardware, and then goes goes to sleep.
;               All the 'work' is done in the interrupt routine
;               
;
;   Input:      None.
;
;   Output:     N/A
;
;*****************************************************************************        
    
    ORG    0
    goto	startClock
    
    ORG    4
    goto	intCode    
    
startClock
	call	initHardware
	
main	
	sleep
	nop
	nop
    goto    main





;*****************************************************************************        
;
;   Function : initHardware
;              Configures the speed of the on-board oscillator,
;              sets up the I/O ports and initialises the LCD
;
;   Input:     None.
;
;*****************************************************************************        
initHardware
	
	; Set oscillator to 4MHz.
	bcf		STATUS, RP1
	bsf		STATUS, RP0
	movlw	0x60
	movwf	OSCCON

	; Wait for the oscillator to stabilse (this is very quick)
ih001
	btfss	OSCCON, HTS
	goto	ih001
	
	
	; Enable interrupt on change on RB5, so we can generate a periodic interrupt
	bsf		IOCB, IOCB5
	
	
	; test - enable some debug output
	clrf	TRISD
	
	; return to our default data bank of 0
	bcf		STATUS, RP0
	
	clrf	timer1s
	clrf	timer60s
	clrf	HH
	clrf	HL
	clrf	MH
	clrf	ML

	
	; a bit of test code - initialise the time to 1:20
	movlw	0x00
	movwf	HH
	movlw	0x01
	movwf	HL
	movlw	0x02
	movwf	MH
	movlw	0x00
	movwf	ML

	
	; Enable PORTB interrupts
	bsf		INTCON, RBIE
	movfw	PORTB
	bcf		INTCON, RBIF
	bsf		INTCON, GIE
	
	; start the low power oscillator on timer1
	bsf		T1CON, T1OSCEN

	; select data bank 2	
	bcf		STATUS, RP0
	bsf		STATUS, RP1

	; Select the frame clock prescale value - 7, to give 1024/32 = 32Hz
	movlw	0x07
	movwf	LCDPS
	
	; configure the segment drivers we need
	movlw	0xFF
	movwf	LCDSE0
	movwf	LCDSE1
	movwf	LCDSE2
	
    ; Select clock source - T1OSC/32
	movlw	0x04
	movwf	LCDCON
	
	; enabled the LCD bias volatges ( maybe not for static? )
	; and enable the LCD output
	bsf		LCDCON, VLCDEN
	bsf		LCDCON, LCDEN
	
	; returned to default memory bank
	bcf		STATUS, RP1
	
	return    

 

;*****************************************************************************        
;
;   Function :  Interrupt vector
;               Occurrs 64 times per second.
;               
;
;   Input:      None.
;
;   Output:     N/A
;
;*****************************************************************************        
intCode
	MOVWF	W_TEMP			;Copy W to TEMP register
	SWAPF	STATUS,W		;Swap status to be saved into W 
	CLRF	STATUS			;bank 0, regardless of current bank, Clears IRP,RP1,RP0
	MOVWF	STATUS_TEMP		;Save status to bank zero STATUS_TEMP register

	movfw	PORTB 
	bcf		INTCON, RBIF
	
	incf	timer1s, F
	movfw	timer1s
	sublw	D'64'
	btfss	STATUS, Z
	goto	ic001
	
	; The one second timer has expired
	clrf	timer1s

    ; Toggle the LCD display bit that controls the : character on the LCD
	call	LCDflash1s
	
    ; Display the current time: update the 4th digit
	call    LCDDigit4Char
	
    ; Display the current time: update the 3rd digit
	call	LCDDigit3Char
	bsf		STATUS, RP1
	movwf	tmp
	movlw	0x80
	andwf	LCDDATA2, F	
	movfw	tmp
	addwf	LCDDATA2, F
	bcf		STATUS, RP1

    ; Display the current time: update the 2nd digit
	call	LCDDigit2Char
	bsf		STATUS, RP1
	movwf	tmp
	movlw	0x80
	andwf	LCDDATA1, F	
	movfw	tmp
	addwf	LCDDATA1, F
	bcf		STATUS, RP1

    ; Display the current time: update the 1st digit
	call	LCDDigit1Char
	bsf		STATUS, RP1
	movwf	tmp
	movlw	0x80
	andwf	LCDDATA0, F	
	movfw	tmp
	addwf	LCDDATA0, F
	bcf		STATUS, RP1
	
    ; Update the 60s counter. Every minute, update the clock variables
	incf	timer60s, F
	movfw	timer60s
	sublw	D'60'
	btfss	STATUS, Z
	goto	ic001
	
	clrf	timer60s
	incf	ML,F
	movfw	ML
	sublw	D'10'
	btfss	STATUS, Z
	goto	ic001
	clrf	ML
	incf	MH,F	
	movfw	MH
	sublw	D'6'
	btfss	STATUS, Z
	goto	ic001
	clrf	MH

	incf	HL,F
	movfw	HL
	sublw	D'10'
	btfss	STATUS, Z
	goto	ic001
	clrf	HL
	incf	HH,F	
	movfw	HH
	sublw	D'2'
	btfss	STATUS, Z
	goto	ic001
	clrf	HH

    ; And thats all. In a real clock application we would have key press
    ; detection to allow the time to be changed, and maybe even add
    ; alarm features. Those are not part of driving an actual LCD, and
    ; are left as an exercise to the reader.

		
ic001
	SWAPF	STATUS_TEMP,W 	;Swap STATUS_TEMP register into W 
	;(sets bank to original state)
	MOVWF	STATUS			;Move W into Status register
	SWAPF	W_TEMP,F		;Swap W_TEMP
	SWAPF	W_TEMP,W		;Swap W_TEMP into W
	retfie



LCDflash1s
	bsf		STATUS, RP1
	movlw	0x80
	xorwf	LCDDATA1, F			
	bcf		STATUS, RP1
	return
	
	
	
LCDDigit4Char	
    movlw   high LCDDigit4Char
    movwf   PCLATH
    movfw   HH
    addwf   PCL,F

	goto	writeD4_0
	goto	writeD4_1

LCDDigit4Char_end

    IF ( (LCDDigit4Char & 0x0FF) >= (LCDDigit4Char_end & 0x0FF) )
        MESSG   "Table LCDDigit4Char overflow"
    ENDIF


writeD4_0
	bsf		STATUS, RP1
	bcf		LCDDATA2,7
	bcf		LCDDATA0,7
	bcf		STATUS, RP1
	return

writeD4_1
	bsf		STATUS, RP1
	bsf		LCDDATA2,7
	bsf		LCDDATA0,7
	bcf		STATUS, RP1
	return


LCDDigit3Char	
    movlw   high LCDDigit3Char
    movwf   PCLATH
    movfw   HL
    addwf   PCL,F

	retlw	0x77	; 0
	retlw	0x41	; 1
	retlw	0x3B	; 2
	retlw	0x6B	; 3
	retlw	0x4D	; 4
	retlw	0x6E	; 5
	retlw	0x7E	; 6
	retlw	0x43	; 7
	retlw	0x7F	; 8
	retlw	0x4F	; 9

LCDDigit3Char_end

    IF ( (LCDDigit3Char & 0x0FF) >= (LCDDigit3Char_end & 0x0FF) )
        MESSG   "Table LCDDigit3Char overflow"
    ENDIF


LCDDigit2Char	
    movlw   high LCDDigit2Char
    movwf   PCLATH
    movfw   MH
    addwf   PCL,F

	retlw	0x3F	; 0
	retlw	0x0C	; 1
	retlw	0x5B	; 2
	retlw	0x5E	; 3
	retlw	0x6C	; 4
	retlw	0x76	; 5
	retlw	0x77	; 6
	retlw	0x1C	; 7
	retlw	0x7F	; 8
	retlw	0x7C	; 9

LCDDigit2Char_end

    IF ( (LCDDigit2Char & 0x0FF) >= (LCDDigit2Char_end & 0x0FF) )
        MESSG   "Table LCDDigit2Char overflow"
    ENDIF


LCDDigit1Char	
    movlw   high LCDDigit1Char
    movwf   PCLATH
    movfw   ML
    addwf   PCL,F

	retlw	0x3F	; 0
	retlw	0x0C	; 1
	retlw	0x5B	; 2
	retlw	0x5E	; 3
	retlw	0x6C	; 4
	retlw	0x76	; 5
	retlw	0x77	; 6
	retlw	0x1C	; 7
	retlw	0x7F	; 8
	retlw	0x7C	; 9

LCDDigit1Char_end

    IF ( (LCDDigit1Char & 0x0FF) >= (LCDDigit1Char_end & 0x0FF) )
        MESSG   "Table LCDDigit1Char overflow"
    ENDIF



;*****************************************************************************        
;
;   Function : getKey
;              Waits for a key to be pressed, and then released
;              The key is on RC0
;
;   Input:     None.
;
;*****************************************************************************        
getKey
	; wait for key to be pressed ( a zero )
	btfsc	PORTC, 0
	goto	getKey
	; delay, to check to see if it is really pressed
    movlw   D'2'
    call    UIWait10ms
	btfsc	PORTC, 0
	goto	getKey

	; now wait for it to be released
waitRelease
	btfss	PORTC, 0
	goto	waitRelease
	; delay, to check to see if it is really pressed
    movlw   D'2'
    call    UIWait10ms
	btfss	PORTC, 0
	goto	waitRelease
	
	return
	

 
;*****************************************************************************        
;
;   Function : UIWait10ms
;              delays for a multiple of 10ms
;
;   Input:     multiple in W
;
;*****************************************************************************        
UIWait10ms
    movwf   delay3 
    
d1ms001
    movlw   D'200'
    call    UIWait50us
    decfsz  delay3, F 
    goto    d1ms001
    return



;*****************************************************************************        
;
;   Function : UIWait50us
;              delays for a multiple of 50us
;
;   Input:     multiple in W
;              cycles are 2 + 1 + (delay2)*( 1+1 + ((d-1)*3 + 2) + 3)  + 3
;              Need 500 cycles at 40MHz : delay1 = 165
;              Need 100 cycles at 8MHz: delay1 = 32
;                         
;
;*****************************************************************************        
UIWait50us             ; 2     ( the call itself )
    movwf   delay2      ; 1  

d1us002
    movlw   D'32'      ; 1          
    movwf   delay1      ; 1          
    
d1us001      
    decfsz  delay1, F   ; 1 or 2        
    goto    d1us001     ; 2               
    
    decfsz  delay2, F   ; 1 or 2
    goto    d1us002     ; 2
    return              ; 2        




    
	; Give some defaults settings for the config registers 
	; MCLR pin is input, internal oscillator with IO enabled,
	; No watch dog, startup timer enabled.
	__config 0x30D4      

; Declare the locations for variables we use in this
; program


timer1s		EQU	0x20	; Counts 32Hz interrupts to give 1s
HH			EQU	0x21
HL			EQU	0x22
MH			EQU	0x23
ML			EQU	0x24
tmp			EQU	0x25	
timer60s	EQU	0x26

STATUS_TEMP EQU 0x7B
W_TEMP		EQU	0x7C
delay1      EQU 0x7D
delay2      EQU 0x7E
delay3      EQU 0x7F
    
    END                ; End of program
    