	Title "Program to demonstrate PLL " 
;****** ******* ******* ******* ******* ******* ******* *** * 
;COPYRIGHT (C) 2006 - Keith Anderson 
;------ ------- ------- ------- ------- ------- ------- --- - 

;****** ******* ******* ******* ******* ******* ******* *** * 
;PllDemo.asm 
;------ ------- ------- ------- ------- ------- ------- --- - 

;****** ******* ******* ******* ******* ******* ******* *** * 
;CONFIGURATION 
;------ ------- ------- ------- ------- ------- ------- --- - 

;CONFIG<13,12> - not used 
;CONFIG<11> - fail safe clock monitor - off 
MyConfig set _FCMEN_OFF 
;CONFIG<10> - Internal External switchover - off 
MyConfig set MyConfig & _IESO_OFF 
;CONFIG<9,8> - brownout detect - on 
MyConfig set MyConfig & _BOD_ON 
;CONFIG<7> - EEPROM code protect - off 
MyConfig set MyConfig & _CPD_OFF 
;CONFIG<6> - Program code protect - off 
MyConfig set MyConfig & _CP_OFF 
;CONFIG<5> - master clear - off 
MyConfig set MyConfig & _MCLRE_OFF 
;CONFIG<4> - power up timer - on 
MyConfig set MyConfig & _PWRTE_ON 
;CONFIG<3> - watchdog - off 
MyConfig set MyConfig & _WDT_OFF 
;CONFIG<2, 1,0> - oscillator - internal, no clock out 
MyConfig set MyConfig & _INTRC_OSC_NOCLKOUT 
	__CONFIG MyConfig 

;****** ******* ******* ******* ******* ******* ******* *** * 
;DEFINE ENVIRONMENT  
;------ ------- ------- ------- ------- ------- ------- --- - 

	list p=12f683 
	include <p12f683.inc> 

	ERRORLEVEL -302, -306 

;****** ******* ******* ******* ******* ******* ******* *** * 
;Fosc 
;------ ------- ------- ------- ------- ------- ------- --- - 

;Define the speed of the clock. 
;It is important for this definition to match 
;the hardware, and that must be done manually. 
Fosc equ 8 ;MHz 

;****** ******* ******* ******* ******* ******* ******* *** * 
;CONSTANTS, STARTUP 
;------ ------- ------- ------- ------- ------- ------- --- - 

Bank0Begin equ 0x0020 
Bank0End equ 0x0080 
Bank1Begin equ 0x00A0 
Bank1End equ 0x00C0 

;****** ******* ******* ******* ******* ******* ******* *** * 
;CONSTANTS, STARTUP, ADCON0 
;------ ------- ------- ------- ------- ------- ------- --- - 

InitADCON0 set 0 

;On Power up, ADCON0 is set to 0x00 

;ADFM sets the ADC result format. 
;0x00 = 0... = Left justified. 
;0x80 = 1... = Right justified. 
;I want right justified. 
InitADCON0 set InitADCON0|(1<<ADFM)

;VCFG sets the reference.
;0x00 = 0... = Internal reference, Vdd
;0x40 = 1... = External reference, GP1/AN1/Vref
;I want Vdd. 
;InitADCON0 set InitADCON0|(1<<VCFG)

;CHS1:CHS0 select the current ADC channel. 
;This is often set dynamically, at run time. 
;0x00 = ...000... = Channel 0, RA0/AN0 
;0x04 = ...001... = Channel 1, RA1/AN1 
;0x08 = ...010... = Channel 2, RA2/AN2 
;0x0C = ...011... = Channel 3, RA3/AN3 
;For this application, I need only one ADC input 
;and I can probably use any ADC channel, 
;but I also use in circuit serial programming, ICSP, 
;and the ADC filter includes a 1u capacitor.  
;Although I isolate this with a 68k resistor, 
;I can probably avoid some red herrings if I use 
;AN2 rather than AN0 or AN1 for the ADC input.
;The value below must be consistent with 
;the hardware and with InitTris.
;This consistency must be maintained manually.
;I can set CHS2:CHS0 statically. 
InitADCON0 set InitADCON0|0x08 

;ADCON0,GO is always set dynamically. 
;InitADCON0 set InitADCON0|(1<<ADGO)

;ADCON0,ADON is set to turn the ADC on. 
;Often it is set dynamically.
;For this application, I use the ADC for the PLL, 
;and I need to set ADCON0,ADON statically. 
InitADCON0 set InitADCON0|(1<<ADON)

;****** ******* ******* ******* ******* ******* ******* *** * 
;CONSTANTS, STARTUP, ANSEL 
;------ ------- ------- ------- ------- ------- ------- --- - 

InitANSEL set 0 

;On Power up, ANSEL is set to 0x0F 

;ADCS2:ADCS0 set the ADC clock. 
;0x00 = 000... = Fosc/2 
;0x10 = 001... = Fosc/8 
;0x20 = 010... = Fosc/32 
;0x30 = x11... = Internal RC clock ~= 250KHz 
;0x40 = 000... = Fosc/4 
;0x50 = 001... = Fosc/16 
;0x60 = 010... = Fosc/64 
;For 8MHz internal oscillator, 
;from the charts in the data sheets, 
;the optimum is Fosc/32. 
InitANSEL set InitANSEL|0x20 

;ANS3:ANS0 set the port configuration. 
;0 = digital.
;1 = analog.
;I want GP2/AN2 analog, and all others digital. 
InitANSEL set InitANSEL|0x04 

;****** ******* ******* ******* ******* ******* ******* *** * 
;CONSTANTS, STARTUP, CCP1CON 
;------ ------- ------- ------- ------- ------- ------- --- - 

InitCCP1CON set 0 

;On Power up, CCP1CON is set to 0x00 

;Bits 7:6 are unused. 
;Bits 5:4 are unused in compare mode. 
;Bits 3:0 are coded: 
;0x00 Off 
;0x01 Undefined 
;0x02 Undefined 
;0x03 Undefined 
;0x04 Capture, every falling edge 
;0x05 Capture, every rising edge 
;0x06 Capture, every 4th rising edge 
;0x07 Capture, every 16th rising edge 
;0x08 Compare, set GP2/CCP1, set PIR2,CCP1IF 
;0x09 Compare, clear GP2/CCP1, set PIR2,CCP1IF 
;0x0A Compare, retain GP2/CCP1, set PIR2,CCP1IF 
;0x0B Compare, retain GP2/CCP1, set PIR2,CCP1IF, 
;	trigger special event, 
;	clear TMR1, start ADC conversion 
;0x0C PWM 
;0x0D PWM 
;0x0E PWM 
;0x0F PWM 
;I need the special event. 
InitCCP1CON set InitCCP1CON|0x0B 

;****** ******* ******* ******* ******* ******* ******* *** * 
;CONSTANTS, STARTUP, CCPR 
;------ ------- ------- ------- ------- ------- ------- --- - 

;It is useful to let the assembler calculate 
;the values for CCPR. 

;****** ******* ******* ******* ******* ******* ******* *** * 
;MACRO TO DEFINE THE TIME(S) TO PRESET INTO CCPR 
;------ ------- ------- ------- ------- ------- ------- --- - 

;TMR1 counts from zero towards CCPR, 
;and produces interrupt on match. 

;The sanity check calculations can be examined 
;manually in the listing and help to ensure 
;that the values are correct. 

;Usually, values for CCPR are needed 
;for several frequencies. 
;Each CCPR is computed double precision, 
;and LOW and HIGH assembler directives should be used 
;where appropriate to extract CCPRL and CCPRH. 

DefineCCPR MACRO ThisFrequency,ThisCCPR 

ThisCCPR set 1000*1000*Fosc/ThisFrequency/4/16 
ThisCCPRH set HIGH ThisCCPR 
ThisCCPRL set LOW ThisCCPR 
SanityCheck set ThisCCPRH*256+ThisCCPRL 
SanityCheck set 1000*1000*Fosc/SanityCheck/4/16 

	ENDM 

;****** ******* ******* ******* ******* ******* ******* *** * 
;CONSTANTS, STARTUP, CCPR, continued 
;------ ------- ------- ------- ------- ------- ------- --- - 

	DefineCCPR 45,CCPR45 
	DefineCCPR 48,CCPR48 
	DefineCCPR 50,CCPR50 
	DefineCCPR 52,CCPR52 
	DefineCCPR 55,CCPR55 
	DefineCCPR 58,CCPR58 
	DefineCCPR 60,CCPR60 
	DefineCCPR 62,CCPR62 
	DefineCCPR 65,CCPR65 

;****** ******* ******* ******* ******* ******* ******* *** * 
;CONSTANTS, CMCON0 
;------ ------- ------- ------- ------- ------- ------- --- - 

InitCMCON0 set 0 

;On Power up, CMCON0 is set to 0x00 
;The comparator is reset, but not off. 
;It must be turned off when GP1 and GP2 are digital.  

;The codes for the comparator are arcane and 
;must be discovered from the table in the data sheets.
InitCMCON0 set InitCMCON0|0x07

;****** ******* ******* ******* ******* ******* ******* *** * 
;CONSTANTS, CMCON1 
;------ ------- ------- ------- ------- ------- ------- --- - 

;InitCMCON1 set 0 

;On Power up, CMCON1 is set to 0x02 

;For this application, the comparator is not used, 
;so for CMCON1, the default is appropriate. 

;****** ******* ******* ******* ******* ******* ******* *** * 
;CONSTANTS, VRCON 
;------ ------- ------- ------- ------- ------- ------- --- - 

;InitVRCON set 0 

;On Power up, VRCON is set to 0x00 

;For this application, the comparator is not used, 
;so for VRCON, the default is appropriate. 

;****** ******* ******* ******* ******* ******* ******* *** * 
;CONSTANTS, INTCON 
;------ ------- ------- ------- ------- ------- ------- --- - 

InitINTCON set 0 

;On Power up, INTCON is set to 0x00 

;Need GIE on for interrupt 
InitINTCON set InitINTCON|(1<<GIE) 

;Need PEIE too for ADC 
InitINTCON set InitINTCON|(1<<PEIE) 

;TMR0 is not used. 
;InitINTCON set InitINTCON|(1<<T0IE) 

;GP2 interrupt is not used. 
;InitINTCON set InitINTCON|(1<<INTE) 

;GP port change interrupt is not used. 
;InitINTCON set InitINTCON|(1<<GPIE) 

;TMR0 is not used. 
;InitINTCON set InitINTCON|(1<<T0IF) 

;GP2 interrupt is not used. 
;InitINTCON set InitINTCON|(1<<INTF) 

;GP port change interrupt is not used. 
;InitINTCON set InitINTCON|(1<<GPIF) 

;****** ******* ******* ******* ******* ******* ******* *** * 
;CONSTANTS, OPTION 
;------ ------- ------- ------- ------- ------- ------- --- - 

InitOption set 0 

;On Power up, OPTION_REG is set to 0xFF 

;A weak pullup on GPIO,3, the input only pin, would 
;be useful, but this pin doesn't have a weak pullup.
;The ADC input pin, GPIO,0 is an analog input 
;and weak pullups are disabled.
;All other pins are outputs.
;Consequently, it is simplest to disable all weak pullups.
InitOption set InitOption|(1<<NOT_GPPU)

;I use GP2 for output, and INTEDG is unimportant.
;InitOption set InitOption|(1<<INTEDG)

;TMR0 is not used. 
;InitOption set InitOption|(1<<T0CS)

;TMR0 is not used, 
;so clock edge is not important. 
;InitOption set InitOption|(1<<T0SE)

;The WDT doesn't need the prescaler. 
;InitOption set InitOption|(1<<PSA)

;For a 10MHz crystal, 
;the prescaler delivers the following times: 
;Prescale TMR0   WDT    Res      Range 
;0        1:2    1:1    800n     204800n 
;1        1:4    1:2    1600n    409600n 
;2        1:8    1:4    3200n    819200n 
;3        1:16   1:8    6400n    1638400n 
;4        1:32   1:16   12800n   3276800n 
;5        1:64   1:32   25600n   6553600n 
;6        1:128  1:64   51200n   13107200n 
;7        1:256  1:128  102400n  26214400n 
;TMR0 is not used, so the prescale is unimportant, 
;and the values have not been recalculated 
;for Fosc = 8MHz. 
;InitOption set InitOption|0x06 

;****** ******* ******* ******* ******* ******* ******* *** * 
;CONSTANTS, PIE1 
;------ ------- ------- ------- ------- ------- ------- --- - 

InitPIE1 set 0 

;On Power up, PIE1 is set to 0x00 

;EE Interrupt is not used.
;InitPIE1 set InitPIE1|(1<<EEIE) 

;ADC interrupt is the primary system tick. 
InitPIE1 set InitPIE1|(1<<ADIE) 

;CCP1 is used, but does not interrupt directly. 
;See ADIE, above. 
;InitPIE1 set InitPIE1|(1<<CCP1IE) 

;The comparator is not used. 
;InitPIE1 set InitPIE1|(1<<CMIE) 

;Oscillator fail interrupt is not used. 
;InitPIE1 set InitPIE1|(1<<OSFIE) 

;TMR2 is not used. 
;InitPIE1 set InitPIE1|(1<<TMR2IE) 

;TMR1 is used, but does not interrupt directly. 
;Instead, TMR1 == CCPR causes a special event trigger, 
;which causes an ADC conversion, 
;and the interrupt is generated when that is done. 
;InitPIE1 set InitPIE1|(1<<TMR1IE) 

;****** ******* ******* ******* ******* ******* ******* *** * 
;CONSTANTS, T1CON 
;------ ------- ------- ------- ------- ------- ------- --- - 

InitT1CON set 0 

;On Power up, T1CON is set to 0x00

;I don't use the gate control for TMR1,
;so gate invert is unimportant. 
;InitT1CON set InitT1CON|(1<<T1GINV)

;I don't use the gate control for TMR1,
;so I need gate enable = 0. 
;InitT1CON set InitT1CON|(1<<TMR1GE)

;Prescale Ratio 
;0x00 1:1 
;0x10 1:2 
;0x20 1:4 
;0x30 1:8 
;Use TMR1 prescale = 0 == 1:1 
InitT1CON set InitT1CON|0

;I use Fosc/4, not the special TMR1 oscillator. 
;InitT1CON set InitT1CON|(1<<T1OSCEN)

;I use Fosc/4, 
;so external clock synchronization is unimportant. 
;InitT1CON set InitT1CON|(1<<T1SYNC)

;I use Fosc/4. 
;InitT1CON set InitT1CON|(1<<TMR1CS)

;I want TMR1 ON. 
InitT1CON set InitT1CON|(1<<TMR1ON)

;****** ******* ******* ******* ******* ******* ******* *** * 
;CONSTANTS, TRIS, WPU, IOC 
;------ ------- ------- ------- ------- ------- ------- --- - 

InitTris set 0 

;On Power up, TRISIO is set to 0x3F 

;GP0 is VcoP,
;a square wave in phase with Vac and lagging Vadc by Pi/2.
;InitTris set InitTris|(1<<0)
VcoP equ 0

;GP1 is VcoQ,
;a square wave lagging VcoP by Pi/2.
;InitTris set InitTris|(1<<1) 
VcoQ equ 1

;GP2 is input for the ADC.
;For this application, I need only one ADC input 
;and I can probably use any ADC channel, 
;but I also use in circuit serial programming, ICSP, 
;and the ADC filter includes a 1u capacitor.  
;Although I isolate this with a 68k resistor, 
;I can probably avoid some red herrings if I use 
;AN2 rather than AN0 or AN1 for the ADC input.
;The value below must be consistent with 
;the hardware and with InitADCON.
;This consistency must be maintained manually.
InitTris set InitTris|(1<<2)

;GP3 is always input, unused in this application.
InitTris set InitTris|(1<<3)

;GP4 is used to reveal interrupt timing.
;InitTris set InitTris|(1<<4) 
PllTime equ 4

;GP5 is lock detect,
;low if LoopCount <> 0,
;high if LoopCount = 0.
;InitTris set InitTris|(1<<5) 
PllLock equ 5 

;****** ******* ******* ******* ******* ******* ******* *** * 
;CONSTANTS, WPU 
;------ ------- ------- ------- ------- ------- ------- --- - 

;InitWPU set 0

;On Power up, WPU is set to 0x37

;For this application, pullups are not used 
;and this is controlled by OPTION_REG,GPUU, 
;so for WPU, the default is appropriate. 

;****** ******* ******* ******* ******* ******* ******* *** * 
;CONSTANTS, IOC 
;------ ------- ------- ------- ------- ------- ------- --- - 

;InitIOC set 0

;On Power up, IOC is set to 0x00

;For this application, interrupt on change is not used, 
;so for IOC, the default is appropriate. 

;****** ******* ******* ******* ******* ******* ******* *** * 
;MACROS 
;------ ------- ------- ------- ------- ------- ------- --- - 

;****** ******* ******* ******* ******* ******* ******* *** * 
;CLEAR BANK 
;------ ------- ------- ------- ------- ------- ------- --- - 

ClearBank MACRO ThisBankBegin,ThisBankEnd 
	LOCAL CB100 

	BankSelect ThisBankBegin 
	movlw LOW ThisBankBegin 
	movwf FSR 
	movlw ThisBankEnd-ThisBankBegin-1 
	movwf ThisBankBegin 

CB100 
;Don't clear the counter in BankBegin. 
	incf FSR,f 
	clrf INDF 
	decfsz ThisBankBegin,f 
	goto CB100 

	ENDM 

;****** ******* ******* ******* ******* ******* ******* *** *
;BankSelect
;BankChange
;------ ------- ------- ------- ------- ------- ------- --- -

;Two macros to set or clear STATUS,RP0.

;If the current bank is uncertain, use 
;	BankSelect	NewVariable
;to generate one instruction to set or clear STATUS,RP0.

;If the current bank is known, usually by knowing 
;the most recently accessed variable in that bank, use
;	BankChange	OldVariable,NewVariable
;to generate one instruction to set or clear STATUS,RP0,
;but only if needs to change.

;Warning!
;BankChange does not check if OldVariable really is 
;in the current bank. 

;There is one special case.
;If the appropriate variable is not known specifically, 
;but is known to be in bank 0, 
;then it is appropriate to use the value 0.
;This is not general, 
;and can not be extended to banks above 0.

;****** ******* ******* ******* ******* ******* ******* *** *
;BankSelect
;------ ------- ------- ------- ------- ------- ------- --- -

BankSelect MACRO NewVariable
	LOCAL NewRP0

NewRP0 set NewVariable&0x0080

	if NewRP0 == 0
	 bcf STATUS,RP0
	else
	 bsf STATUS,RP0
	endif

	ENDM

;****** ******* ******* ******* ******* ******* ******* *** *
;BankChange
;------ ------- ------- ------- ------- ------- ------- --- -

BankChange	MACRO	OldVariable,NewVariable
	LOCAL	OldRP0,NewRP0

OldRP0 set OldVariable&0x0080
NewRP0 set NewVariable&0x0080

	if NewRP0 == 0 && OldRP0 != 0
	 bcf STATUS,RP0
	endif
	if NewRP0 != 0 && OldRP0 == 0
	 bsf STATUS,RP0
	endif

	ENDM

;****** ******* ******* ******* ******* ******* ******* *** * 
;INCLUDE FILES 
;------ ------- ------- ------- ------- ------- ------- --- - 

	include <IfMnemonics.asm> 
	include <If16.asm> 
	include <MathAddSub16.asm> 
	include <MathIncDecModK8.asm> 
	include <MathDecToZ8.asm> 
	include <MathMove16.asm> 
	include <MathShift16.asm> 
	include <SystemContext.asm> 
	include <SystemTables.asm> 

;****** ******* ******* ******* ******* ******* ******* *** * 
;VARIABLES 
;------ ------- ------- ------- ------- ------- ------- --- - 

	org 0x20

;****** ******* ******* ******* ******* ******* ******* *** * 
;VARIABLES, INTERRUPT 
;------ ------- ------- ------- ------- ------- ------- --- - 

;The PLL strategy matches TMR1 to the input frequency. 

;CCP2 is adjusted to provide 16 counts of TMR1 in each cycle. 

;TickCount counts each tick within each cycle. 
;Because of the heavy filtering, the signal at 
;the ADC input lags the signal at the AC input 
;by 90 degrees. 
;Consequently, TickCount = 0 corresponds to 
;the maximum of the ADC input 
;and a zero crossing of AC input. 

;When the loop is locked, the angle and action(s) 
;corresponding to each value of TickCount are: 
;TickCount Angle Action 
;0         90    PreviousCycleTot = LatestCycleTot 
;	           LatestCycleTot = 0 
;1         112.5 LatestCycleTot = LatestCycleTot +AdcTemp 
;2         135   LatestCycleTot = LatestCycleTot +AdcTemp 
;3         157.5 LatestCycleTot = LatestCycleTot +AdcTemp 
;4         180   LatestCycleTot = LatestCycleTot +AdcTemp 
;5         202.5 LatestCycleTot = LatestCycleTot +AdcTemp 
;6         225   LatestCycleTot = LatestCycleTot +AdcTemp 
;7         247.5 LatestCycleTot = LatestCycleTot +AdcTemp 
;8         270   LatestCycleTot = LatestCycleTot  
;9         292.5 LatestCycleTot = LatestCycleTot -AdcTemp 
;10        315   LatestCycleTot = LatestCycleTot -AdcTemp 
;11        337.5 LatestCycleTot = LatestCycleTot -AdcTemp 
;12        360   LatestCycleTot = LatestCycleTot -AdcTemp 
;13        382.5 LatestCycleTot = LatestCycleTot -AdcTemp 
;14        405   LatestCycleTot = LatestCycleTot -AdcTemp 
;15        427.5 LatestCycleTot = LatestCycleTot -AdcTemp 
;	           Update CCP2 

TickCount res 1 

;Working space for the AC power voltage read from the ADC. 
AdcTemp res 2 

;The PLL filter variables. 
LatestCycleTot res 2 
PreviousCycleTot res 2 

;It is useful to be able to detect if the loop is locked. 
;The strategy is: 
;The PLL calculates a measure of the phase error. 
;If this measure is suitably small, 
;then the loop is probably locked. 
;If it remains suitably small for several 
;consecutive cycles, then the probability 
;that the loop is locked increases. 

;Specifically, 
;If the error is less than LoopError1K, 
;then decrement LoopCount, 
;else reset LoopCount to LoopCountK. 

;Currently, I use only one value of LoopError1K 
;and use it only to detect initial lock.  
;In this case, I can assume that the input signal 
;will be very steady, or wait until it is so, 
;and consequently I can make LoopError1K very small. 

;A future enhancement might be to detect loss of lock, 
;but in that case, I will probably need another 
;value, somewhat larger than LoopError1K, 
;so that it is large enough to allow CCPR2 
;to vary enough to keep the loop locked.

;The calculation of LoopError2K below is a hint of 
;a suitable value.  This is a rather generous 
;limit, corresponding to an error of about 1Hz.

;Because the values CCPxx are calculated from 
;the reciprocal of Fosc, in the calculation below, 
;CCPR52 is numerically smaller than CCPR48.

LoopCount res 1 
LoopCountK equ 30 

;Although the phase error is useful to the PIC, 
;it isn't wonderfully meaningful to humans, 
;and the size of the error is more meaningful 
;if considered in terms of the change to CCPR.
;ErrorToCCPRShift is used to convert the scale 
;of the error from CCPR units to phase error units.

;See later, near the relevant shift instruction, 
;for more explanation of the calculation 
;of ErrorToCCPRShift.
;MPLAB needs it to be defined here.

;To implement b = 32, 
;the appropriate shift is the nett result of: 
;Right 1 to undo the scale used while calculating b and c,
;Left 5 to implement b = 32,
;Right 9 to undo the scale of 512 implied by the ADC.

;Use comments to leave one and only one 
;of the statements below active.
;ErrorToCCPRShift equ 1-4+9 ;b = 16
ErrorToCCPRShift equ 1-5+9 ;b = 32
;ErrorToCCPRShift equ 1-6+9 ;b = 64

LoopError1K equ 4*(1<<ErrorToCCPRShift) 
LoopError2K equ ((CCPR48-CCPR52)/4)*(1<<ErrorToCCPRShift) 

;****** ******* ******* ******* ******* ******* ******* *** * 
;VARIABLES, COMMON  
;------ ------- ------- ------- ------- ------- ------- --- - 

	org 0x70

;****** ******* ******* ******* ******* ******* ******* *** * 
;VARIABLES, COMMON, INTERRUPT CONTEXT 
;------ ------- ------- ------- ------- ------- ------- --- - 

;The top 16 bytes are common to all banks 
;and are ideal for saving context.

;The SaveContext and RestoreContext macros depend upon 
;using the names below to define the variables 
;used to save context.
;TempW and TempSTATUS must be defined always.
;TempPCLATH and/or TempFSR may be left undefined, 
;in which case, they will not be included in 
;the Save/Restore strategy.
;Refer to the notes accompanying these macros 
;for more information.

TempW res 1
TempStatus res 1
TempPCLATH res 1
TempFSR res 1

;General registers used inside interrupt 
;must be separate from those used outside interrupt. 
;For the PIC architecture, this is more efficient 
;that finding somewhere to save registers. 

IW0 res 1 
IW1 res 1 

;****** ******* ******* ******* ******* ******* ******* *** * 
;VARIABLES, COMMON, GENERAL REGISTERS 
;------ ------- ------- ------- ------- ------- ------- --- - 

;It isn't easy to develop a uniform strategy to manage 
;argument passing and bank management. 

;Single scalars can be passed in w. 
;A few scalars can be passed 
;in the common unbanked registers. 
;There is no easy way to pass many scalars or vectors. 

;For this application, outside interrupt, 
;four of the common unbanked registers are used 
;for argument passing and as general temporary variables. 

WW0 res 1 
WW1 res 1 
WW2 res 1 
WW3 res 1 

;****** ******* ******* ******* ******* ******* ******* *** * 
;VECTORS 
;------ ------- ------- ------- ------- ------- ------- --- - 

;Reset
	org 0x00

;I want to be able to put Startup on any suitable page. 
	goto Startup 

;Interrupt
	org 0x04

;A subtlty of the PIC is that it is not a good idea 
;to have any goto statements in the interrupt routine 
;until PCLATH has been saved. 

	SaveContext 

	goto Interrupt 

;****** ******* ******* ******* ******* ******* ******* *** * 
;LOOKUP AND/OR JUMP TABLES 
;------ ------- ------- ------- ------- ------- ------- --- - 

	org 0x0010

;****** ******* ******* ******* ******* ******* ******* *** * 
;LOOKUP TABLES 
;------ ------- ------- ------- ------- ------- ------- --- - 

;	None 

;****** ******* ******* ******* ******* ******* ******* *** * 
;JUMP TABLES, PLLState 
;------ ------- ------- ------- ------- ------- ------- --- - 

;Although allocating a specific state to process 
;each ADC sample might seem a little extravagant, 
;this strategy allows application specific actions 
;to be scheduled easily and reliably.
;A more efficient strategy invites bugs to 
;party, party, party, and the temptation 
;to search for one should be resisted.
  
PLLStateStart 

	addwf PCL,f 
	goto PLL00 
	goto PLL01 
	goto PLL02 
	goto PLL03 
	goto PLL04 
	goto PLL05 
	goto PLL06 
	goto PLL07 
	goto PLL08 
	goto PLL09 
	goto PLL10 
	goto PLL11 
	goto PLL12 
	goto PLL13 
	goto PLL14 
	goto PLL15 

PLLStateEnd 

;****** ******* ******* ******* ******* ******* ******* *** * 
;PROCEDURES 
;------ ------- ------- ------- ------- ------- ------- --- - 

	org 0x0100

;****** ******* ******* ******* ******* ******* ******* *** * 
;PROCEDURES, MAIN  
;------ ------- ------- ------- ------- ------- ------- --- - 

Begin 

	goto Begin 

;****** ******* ******* ******* ******* ******* ******* *** * 
;PROCEDURES, INTERRUPT 
;------ ------- ------- ------- ------- ------- ------- --- - 

;****** ******* ******* ******* ******* ******* ******* *** * 
;TABLE RANGE CHECK PROCEDURES 
;------ ------- ------- ------- ------- ------- ------- --- - 

PLLState 
	TableRangeCheck PLLStateStart,PLLStateEnd 

;****** ******* ******* ******* ******* ******* ******* *** * 
;INTERRUPT 
;------ ------- ------- ------- ------- ------- ------- --- - 

Interrupt 

	BankSelect GPIO 
	bsf GPIO,PllTime

;There is only one interrupt enabled: PIR1,ADIF. 
;The action depends upon TickCount. 

	BankSelect PIR1 

;Interrupt not PIR1,ADIF is a serious error, 
;but there is no need to process it explicitly. 
;Just return without patting the watchdog. 
;When the watchdog bites, the PIC will reset. 
	btfss PIR1,ADIF 
	goto IH390 

	bcf PIR1,ADIF 

;Copy ADRES to AdcTemp. 
;It is a little irritating that ADRESH and ADRESL 
;are in different banks. 
;I explored clever use of IW0,IW1, 
;but lost more on the swings 
;than I gained on the roundabouts. 
	BankSelect ADRESH 
	movf ADRESH,w 
	BankSelect AdcTemp 
	movwf AdcTemp 
	BankSelect ADRESL 
	movf ADRESL,w 
	BankSelect AdcTemp 
	movwf AdcTemp+1 

;Pat the watchdog. 
	clrwdt 

;Application specific actions appropriate at every 
;sample time can be inserted here.
;Examples include timers for beeps and/or keypad debounce.

;Perform the specific action for this sample. 
	BankSelect TickCount 
	IncModK8 TickCount,15 
	movf TickCount,w 
	goto PLLState 

PLL00 

	nop

;PreviousCycleTot = LatestCycleTot 
;LatestCycleTot = AdcTemp*EdgeStrategy 

EdgeStrategy equ 0

	MoveVV16 LatestCycleTot,PreviousCycleTot 

	if EdgeStrategy==0 
	clrf LatestCycleTot 
	clrf LatestCycleTot+1 
	endif 

	if EdgeStrategy==1 
	MoveVV16 AdcTemp,LatestCycleTot 
	endif 

	BankSelect GPIO
	bsf GPIO,VcoQ

	goto IH390

PLL01 

	nop

;LatestCycleTot = LatestCycleTot +AdcTemp 
	AddVV16 AdcTemp,LatestCycleTot 

	goto IH390 

PLL02 

	nop

;LatestCycleTot = LatestCycleTot +AdcTemp 
	AddVV16 AdcTemp,LatestCycleTot 

	goto IH390 

PLL03 

	nop

;LatestCycleTot = LatestCycleTot +AdcTemp 
	AddVV16 AdcTemp,LatestCycleTot 

	goto IH390 

PLL04 

	nop

;LatestCycleTot = LatestCycleTot +AdcTemp 
	AddVV16 AdcTemp,LatestCycleTot 

	BankSelect GPIO
	bsf GPIO,VcoP

	goto IH390

PLL05 

	nop

;LatestCycleTot = LatestCycleTot +AdcTemp 
	AddVV16 AdcTemp,LatestCycleTot 

	goto IH390 

PLL06 

	nop

;LatestCycleTot = LatestCycleTot +AdcTemp 
	AddVV16 AdcTemp,LatestCycleTot 

	goto IH390 

PLL07 

	nop

;LatestCycleTot = LatestCycleTot +AdcTemp 
	AddVV16 AdcTemp,LatestCycleTot 

	goto IH390

PLL08 

	nop

;LatestCycleTot = LatestCycleTot -AdcTemp*EdgeStrategy 

	if EdgeStrategy==1 
	SubVV16 AdcTemp,LatestCycleTot 
	endif 

	BankSelect GPIO
	bcf GPIO,VcoQ

	goto IH390

PLL09 

	nop

;LatestCycleTot = LatestCycleTot -AdcTemp 
	SubVV16 AdcTemp,LatestCycleTot 

	goto IH390 

PLL10 

	nop

;LatestCycleTot = LatestCycleTot -AdcTemp 
	SubVV16 AdcTemp,LatestCycleTot 

	goto IH390 

PLL11 

	nop

;LatestCycleTot = LatestCycleTot -AdcTemp 
	SubVV16 AdcTemp,LatestCycleTot 

	goto IH390 

PLL12 

	nop

;LatestCycleTot = LatestCycleTot -AdcTemp 
	SubVV16 AdcTemp,LatestCycleTot 

	BankSelect GPIO
	bcf GPIO,VcoP

	goto IH390

PLL13 

	nop

;LatestCycleTot = LatestCycleTot -AdcTemp 
	SubVV16 AdcTemp,LatestCycleTot 

	goto IH390 

PLL14 

	nop

;LatestCycleTot = LatestCycleTot -AdcTemp 
	SubVV16 AdcTemp,LatestCycleTot 

	goto IH390 

PLL15 

	nop

;LatestCycleTot = LatestCycleTot -AdcTemp 
;Update CCPR2 
	SubVV16 AdcTemp,LatestCycleTot 

;Update CCP2. 
;The equation is: 
;NewCCP2 = OldCCP2 + b*LatestCycleTot + c*PreviousCycleTot 
;Testing indicates that suitable parameters are: 
;b = -32 
;c = -0.75*b = +24 

;The calculations below scale b and c to optimise 
;the number of shifs needed while maintaining accuracy 
;and avoiding overflow.

;PreviousCycleTot will be replaced by LatestCycleTot 
;and is available for working. 

;Form (3/2)*PreviousCycleTot in PreviousCycleTot. 
;3 = 2+1. 
;3/2 = 1+(1/2)
	MoveVV16 PreviousCycleTot,IW0 
	ShiftRightVK16 PreviousCycleTot,1
	AddVV16 IW0,PreviousCycleTot 

;Form (4/2)*LatestCycelTot in IW0,IW1. 
	MoveVV16 LatestCycleTot,IW0 
	ShiftLeftVK16 IW0,1

;Form (3/2)*PreviousCycleTot - (4/2)*LatestCycleTot 
;in PreviousCycleTot. 
	SubVV16 IW0,PreviousCycleTot 

;The value in PreviousCycleTot 
;is a measure of the phase error. 
;As part of the lock detect strategy, 
;form the absolute value of this error in IW0,IW1. 

	MoveVV16 PreviousCycleTot,IW0 

	AbsV16 IW0 

;If the phase error is too big, 
;then reset LoopCount. 

	IfVK16 IW0,LT,LoopError1K 
	goto PLL15c 

	movlw LoopCountK 
	movwf LoopCount 

PLL15c 

;Decrement LoopCount. 
	DecToZ8 LoopCount 

;I experimented with clever strategies to 
;maximize the information revealed by PllLock, 
;but lost more on the swings 
;than I gained on the roundabouts. 
;To examine details of how well the loop locks, 
;VcoP and VcoQ should be used.  
;VcoQ changes at phases PLL00 and PLL08, 
;VcoP changes at PLL004 and at PLL12.  
;Consequently, the information at VcoQ is 
;a little easier to interpret than that at VcoP.

	movf LoopCount,f

	BankSelect GPIO

	btfsc STATUS,Z
	bsf GPIO,PllLock
	btfss STATUS,Z
	bcf GPIO,PllLock

;Scale b and c. 
;The shift is not at all obvious.

;From Excel, with Vin = 1.0Sin(wt+phi),
;the optimum values are b = 32, c = -24.

;The ADC produces values that range 0 to 1023,
;where nominally, 0 == -1V, 512 == 0V, 1023 == 0.999V.

;The gain of this type of PLL is sensitive 
;to the amplitude of the input voltage, 
;and in practice, the input circuit ensures that 
;the ADC is unlikely to produce values as low as 0 
;or as high as 1023.

;This means that the 1.0V input in Excel 
;corresponds to an ADC deviation of 
;somewhat less than 512.

;Further, the calculations above have implemented 
;the effect of b = (4/2), c = -(3/2).

;Consequently, to implement b = 32, 
;the appropriate shift is the nett result of: 
;Right 1 to undo the 4/2 already implemented,
;Left 5 to implement b = 32,
;Right 9 to undo the scale of 512 implied by the ADC.

;ErrorToCCPRShift equ 1-5+9

	BankSelect PreviousCycleTot

	ShiftRightVK16 PreviousCycleTot,ErrorToCCPRShift

;Get CCPR1 into IW0,IW1. 
	BankSelect CCPR1L 
	movf CCPR1H,w 
	movwf IW0 
	movf CCPR1L,w 
	movwf IW1 

;Form OldCCPR2 -32*LatestCycleTot +24*PreviousCycleTot 
;in IW0,IW1. 

	BankSelect PreviousCycleTot 
	AddVV16 PreviousCycleTot,IW0 

;It is important to constrain CCPR2 to a sensible range. 
	IfVK16 IW0,GT,CCPR62 
	goto PLL15e 
	MoveKV16 CCPR62,IW0 
PLL15e 

	IfVK16 IW0,LT,CCPR48 
	goto PLL15f 
	MoveKV16 CCPR48,IW0 
PLL15f 

;Loading CCPR2 depends upon a trick. 
;If it is just loaded, TMR1 might match prematurely 
;and trigger a spurious ADC conversion and TMR1 reset. 

;Instead, IW0,7 is set so that the value 
;initially loaded will be much bigger 
;than any value expected in TMR1. 

;After both bytes of CCPR2 have been loaded, 
;CCPR2H,7 is cleared. 

;For this application, 
;CCPR2 is not expected to exceed 0x0FFF, 
;so this strategy should be very safe. 

;Update CCPR2 from IW0. 
	BankSelect CCPR1L 
	bsf IW0,7 
	movf IW0,w 
	movwf CCPR1H 
	movf IW1,w 
	movwf CCPR1L 
	bcf CCPR1H,7 

;	goto IH390

IH390 

	BankSelect GPIO 
	bcf GPIO,PllTime

;Restore context and return. 
	RestoreContext 
	retfie 

;****** ******* ******* ******* ******* ******* ******* *** * 
;PLL INITIALIZE 
;------ ------- ------- ------- ------- ------- ------- --- - 

PllInitialize

	BankSelect LoopCount

	movlw LoopCountK
	movwf LoopCount

	BankSelect CCPR1L

	movlw LOW CCPR55
	movwf CCPR1L
	movlw HIGH CCPR55
	movwf CCPR1H

	return

;****** ******* ******* ******* ******* ******* ******* *** * 
;WAIT FOR LOCK 
;------ ------- ------- ------- ------- ------- ------- --- - 

;After power off/on, startup must wait till the loop locks.
;It is unimportant that the 
;	BankSelect
;below is inside the wait loop.

WaitForLock

	BankSelect LoopCount

	movf LoopCount,f
	btfss STATUS,Z
	goto WaitForLock 

;Get CCPR into IW0,IW1. 
	BankSelect CCPR1L 
	movf CCPR1H,w 
	movwf IW0 
	movf CCPR1L,w 
	movwf IW1 

;In the tests below, 
;the constants are derived from 1/Fosc, 
;so CCPR1 in IW0,IW1 > CCPRxx 
;implies that the VCO frequency is less than 
;the reference frequency.

	IfVK16 IW0,GT,CCPR48
	goto WL99
	IfVK16 IW0,GT,CCPR52
	goto WL50
	IfVK16 IW0,GT,CCPR58
	goto WL99
	IfVK16 IW0,GT,CCPR62
	goto WL60

;Neither 50Hz nor 60Hz, error.
WL99
	goto WaitForLock

;50Hz
WL50

;Perform application specific actions 
;appropriate for 50Hz power.
	return
	
;60Hz
WL60
	
;Perform application specific actions 
;appropriate for 60Hz power.
	return

;****** ******* ******* ******* ******* ******* ******* *** * 
;PROCEDURES, STARTUP  
;------ ------- ------- ------- ------- ------- ------- --- - 

;****** ******* ******* ******* ******* ******* ******* *** * 
;STARTUP, INITIALIZE VARIABLES 
;------ ------- ------- ------- ------- ------- ------- --- - 

Startup 

;Most variables need to be cleared. 
	ClearBank Bank0Begin,Bank0End 
	ClearBank Bank1Begin,Bank1End 

;****** ******* ******* ******* ******* ******* ******* *** * 
;STARTUP, PREPARE PLL 
;------ ------- ------- ------- ------- ------- ------- --- - 

	call PllInitialize 

;****** ******* ******* ******* ******* ******* ******* *** * 
;STARTUP, PREPARE SPECIAL REGISTERS 
;------ ------- ------- ------- ------- ------- ------- --- - 

;Bank 0 special registers. 

	BankSelect T1CON 

;INTCON, see later. 

;T1CON 
;The timer seems to like to be setup with TMR1ON = 0, 
;then to have it explicitly set by bsf T1CON,TMR1ON 
	movlw InitT1CON&~(1<<TMR1ON) ;Init timer 
;	andlw ~(1<<TMR1ON) 
	movwf T1CON 
	bsf T1CON,TMR1ON 

;T2CON, not used. 

;CCP1CON 
;See also call to PllInitialize, above. 
	movlw LOW CCPR55
	movwf CCPR1L
	movlw HIGH CCPR55
	movwf CCPR1H
	movlw InitCCP1CON
	movwf	CCP1CON 

;WDTCON, not used. 

;CMCON0 
	movlw InitCMCON0 
	movwf CMCON0 

;CMCON1, use default. 

;ADCON0 
	movlw InitADCON0 
	movwf ADCON0 

;Bank 1 special registers. 

	BankSelect OPTION_REG 

;OPTION 
;See example in PIC manual. 
	clrwdt 
	movlw InitOption 
	movwf OPTION_REG 

;TRIS 
	movlw InitTris 
	movwf TRISIO 

;PIE1 
	movlw InitPIE1 
	movwf PIE1 

;PCON, not used. 

;OSCCON 
;After reset, the clock frequency is 4MHz.
;It is necessary to set IRCF0 
;to change the frequency to 8MHz. 
	bsf OSCCON,IRCF0

;There is little else to do, so I might as well 
;wait here until OSCCON,HTS is set, to indicate 
;that the clock has settled to the new speed.
S100
	btfss OSCCON,HTS
	goto S100

;OSCTUNE, use default. 

;PR2, not used.

;WPU, use default. 

;IOC, use default. 

;VRCON, not used.

;ANSEL 
	movlw InitANSEL 
	movwf ANSEL 

;The clock speed should have changed to 8MHz, 
;and I expect a pulse close to 2.5uS long.

	BankSelect GPIO

	bsf GPIO,0
	nop
	nop
	nop
	nop
	bcf GPIO,0

;****** ******* ******* ******* ******* ******* ******* *** * 
;STARTUP, ENABLE INTERRUPT 
;------ ------- ------- ------- ------- ------- ------- --- - 

;All of the other special registers have been set.
;When I set INTCON, 
;TMR1 will start, starting the ADC, CCP1, and the PLL.

	BankSelect INTCON 

	movlw InitINTCON 
	movwf INTCON 

;****** ******* ******* ******* ******* ******* ******* *** * 
;STARTUP, WAIT FOR LOCK 
;------ ------- ------- ------- ------- ------- ------- --- - 

;After power off/on, 
;I need to discover if the power is 50Hz or 60Hz. 
;The calibrate strategy assumes the following: 
;1/ 
;The power frequency is either 50Hz or 60Hz. 
;2/ 
;The power frequency and the PIC clock each have 
;suitable short term stability. 

;These assumptions are reasonable.

;The power frequency is managed to deliver excellent 
;long term stability. 
;Although some short term deviation is allowed, 
;this rarely exceeds 0.1%. 
;The PIC clock uses a resonator. 
;Although this is not as accurate as a crystal, 
;the total error is unlikely to exceed 0.5%. 

;This strategy can cope with total error 
;of up to 2/60, which is more than 3%. 

;Wait for the loop to lock. 

	call WaitForLock

;The loop has locked. 
;The called procedure does all necessary housekeeping.

;Untill the PLL is debugged, stick here.
;	goto $

	goto Begin 

;****** ******* ******* ******* ******* ******* ******* *** * 
;COPYRIGHT (C) 2006 - Keith Anderson 
;------ ------- ------- ------- ------- ------- ------- --- - 

	end 



