/**********************
K9670 Mega Box Example Code

The following code provides use examples for 
the various features in the Mega Box, focusing 
on the rotary encoder, liquid crystal display
and infrared receiver. The example increments
/decrements a counter on the LCD as the encoder
is turned, additionally PWMing the backlight at 
the counter value, and toggles LEDs on the front 
switches as buttons 1-4 on an IR remote are pressed.

Uses the build in Arduino LiquidCrystal library
and PJRC's Encoder and IR libraries, available from
https://www.pjrc.com

The Altronics A1012 Learning remote has been confirmed 
working with the IR receiver and PJRC's library. 
Code 089 (Philips TV) used.

Code by Tom Skevington 
Last edited: 6/9/17
**********************/

#include <LiquidCrystal.h>
#include <Encoder.h>
#include <IRremote.h>

//PIN CONNECTIONS
  
  //ENCODER
  /*The encoder works best when used on pins with interrupt capability.
  For ATMega328P based Arduinos, Digital pins 2 & 3 both have interrupt 
  capability, so connect A and B from the encoder interface to pins 2 & 3*/
const unsigned char EncA=2, EncB=3;
  //Then declare an encoder instance using the assigned pins
Encoder Enc(EncA,EncB);
  
  //LCD
  /*The LCD requires either 4 or 7 data pins and two more for 'RS' (register select)
  and 'E' (or EN, the Enable signal) and optionally 'BL' (backlight, although this
  can be wired directly to the 5V supply if the ability to adjust the backlight isn't
  needed). To save pins, we connect the LCD in 4-bit mode. There are no special 
  requirements for these pins, although connecting BL to a PWM capable pin allows
  dimming of the backlight*/
const unsigned char RS=4, EN=5, D4=6, D5=7, D6=8, D7=9, BL=10;
  //Then declare an LCD instance using the assigned pins
LiquidCrystal lcd(RS,EN,D4,D5,D6,D7);

  //IR
  /*The IR receiver has no special requirements for the pin it's attached to,
  so pick any spare pin which doesn't have a required secondary function*/
const unsigned char IRPin=11;
  //Then declare an IR instance using the assigned pin
IRrecv IR(IRPin);
  //and a results structure to hold the decoded IR code
decode_results results;
  
  //LEDs
const unsigned char LED1=A3;
const unsigned char LED2=A2;
const unsigned char LED3=A1;
const unsigned char LED4=A0;
//END PIN CONNECTIONS


void setup() {

  lcd.begin(16,2);              //Initalise the LCD for 16 characters and 2 rows
  lcd.clear();
  analogWrite(BL, 255);                  
  lcd.print("K9670 Mega Box");  
  lcd.setCursor(0,1);           //Move cursor to start of bottom line (0 is the index of the top line, and first character of each line)
  lcd.print("Example Code");    
  IR.enableIRIn();              //Enable the IR receiver to begin receiving
  Enc.write(0);                 //Reset the encoder's accumulated position
  pinMode(LED1,OUTPUT);         
  pinMode(LED2,OUTPUT);         
  pinMode(LED3,OUTPUT);         
  pinMode(LED4,OUTPUT);         
  delay(2000);
}

void loop() {

  static int LastPosition=0;           
  int NewPosition=Enc.read();           //Grab the current accumulated encoder value
                                        
  if(NewPosition!=LastPosition) {       
    LastPosition=NewPosition;           
    lcd.clear();                        
    lcd.print(NewPosition);             
    if(NewPosition<256) {               
      if(NewPosition>-1) {              
        analogWrite(BL,NewPosition);    //If the encoder value is within the PWM output range, use that to set the LCD backlight
      }
    }
  } else if(IR.decode(&results)) {      //Grab a decoded IR code if available
    Serial.print("Got IR code: 0x");
    Serial.println(results.value,HEX);
    switch(results.value) {             //Adjust LEDs based on received code
      case 0x001:
        digitalWrite(LED1,HIGH);
        break;
      case 0x002:
        digitalWrite(LED2,HIGH);
        break;
      case 0x003:
        digitalWrite(LED3,HIGH);
        break;
      case 0x004:
        digitalWrite(LED4,HIGH);
        break;
      case 0x801:
        digitalWrite(LED1,LOW);
        break;
      case 0x802:
        digitalWrite(LED2,LOW);
        break;
      case 0x803:
        digitalWrite(LED3,LOW);
        break;
      case 0x804:
        digitalWrite(LED4,LOW);
        break;
    }
    IR.resume();                        //Resume searching for IR transmissions
  } else {
    delay(10);
  }
}
