

#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <unistd.h>
#include <time.h>
#include <string.h>
#include <errno.h>
#include <sys/mman.h>
#include <sys/time.h>
                            
#define GPIO_ADD    0x20200000L // physical address of I/O peripherals on the ARM processor
#define GPIO_SEL0   1           // Offset of SEL register for GPs0..9 into GPIO bank   
#define GPIO_INP    13          // Offset of PIN INPUT value register for GP0..31 into GPIO bank

#define PAGE_SIZE   4096        
#define BLOCK_SIZE  PAGE_SIZE


int  mem_fd     = 0;
char *gpio_mmap = NULL;
char *gpio_ram  = NULL;
volatile unsigned int *gpio = NULL;


/* These 'defines' map the peripheral pin functions to our circuits buttons pins */
/* See Broadcom BCM2835-ARM-Peripherals.pdf 6/2/2012 */

// Set the three pins to inputs
#define PIN_INPUTS    *(gpio+GPIO_SEL0) &= 0xF8FFFFC0L

// register & masks for individual button levels. A high level is "not pressed"
#define LEFT_LEVEL    (*(gpio+GPIO_INP) & 0x00000001L)
#define RIGHT_LEVEL   (*(gpio+GPIO_INP) & 0x00000002L)
#define SELECT_LEVEL  (*(gpio+GPIO_INP) & 0x00000100L)


void setup_io()
{

   /* open /dev/mem to get acess to physical ram */
   if ((mem_fd = open("/dev/mem", O_RDWR|O_SYNC) ) < 0) {
      printf("can't open /dev/mem. Did you run the program with administrator rights?\n");
      exit (-1);
   }

   /* Allocate a block of virtual RAM in our application's address space */
   if ((gpio_ram = malloc(BLOCK_SIZE + (PAGE_SIZE-1))) == NULL) {
      printf("allocation error \n");
      exit (-1);
   }

   /* Make sure the pointer is on 4K boundary */
   if ((unsigned long)gpio_ram % PAGE_SIZE)
     gpio_ram += PAGE_SIZE - ((unsigned long)gpio_ram % PAGE_SIZE);

   /* Now map the physical addresses of the peripheral control registers 
      into our address space */
   gpio_mmap = (unsigned char *)mmap(
      (caddr_t)gpio_ram,
      BLOCK_SIZE,
      PROT_READ|PROT_WRITE,
      MAP_SHARED|MAP_FIXED,
      mem_fd,
      GPIO_ADD
   );

   if ((long)gpio_mmap < 0) {
      printf("unable to map the memory. Did you run the program with administrator rights?\n");
      exit (-1);
   }

   /* Always use a volatile pointer to hardware registers */
   gpio = (volatile unsigned *)gpio_mmap;

   /* Now we have access to the hardware reigsters we can start twiddling I/O pins */

   /* Switch GPIO 0, 1 and 4 to input mode */
   PIN_INPUTS;

   /* Short delay to allow the I/O lines to settle. */
   usleep(2);
}


int main(int argc, char **argv)
{ 
   int retval = -1;
   int left_val,right_val,select_val;

   
   /* Check that the program was called correctly */
   if ( argc > 2 ) {
      printf("Too many arguments specified.\nRun as:\nbtn-pi\nor\nbtn-pi -t\n");
      exit (-1);
   } 

   /* Set up gpi pointer for direct register access */
   setup_io();
      
   if ( argc == 2 ) {
      /* If the number of arguments are two, that means the user enter -t. */
      /* Just return whether a key is pressed (1) or not (0) */

      if (LEFT_LEVEL && RIGHT_LEVEL && SELECT_LEVEL)
         retval = 0;
      else
         retval = 1;
      
   } else {
      /* wait for a key to be pressed, and return it */
      /* '1' for LEFT, '2' for RIGHT and '3' for SELECT */
   
      do {
         /* We take a copy of the register values, as a button may be */
         /* released between it being detected as pressed and determining */
         /* the return value */
         left_val = LEFT_LEVEL;
         right_val = RIGHT_LEVEL;
         select_val = SELECT_LEVEL;
         usleep(100);  // Don't consume all the CPU time in this loop
      } while ( left_val && right_val && select_val) ;

      if (!left_val) retval = 1;
      if (!right_val) retval = 2;
      if (!select_val) retval = 3;

      /* Sleep for 10ms - this is a little debounce */
      usleep(10000);
   }

   return retval;
}
