/****************************************************************************
*
*   Module:     pic24ps2key.c
*   Author:     Mike Hibbett
*   Date:       19/4/09
*   Purpose:    Main source file for the pic24 PS2 keyboard library.
*                
*               This must be compiled for the PIC24HJ128GP202.
*               The processor clock must run at 80Mhz with an Fcy of 40MHz
*
****************************************************************************/

/**** INCLUDES *************************************************************/
#include "p24HJ128GP202.h"
#include <string.h>
#include "pic24ps2key.h"


/**** CONSTANTS ************************************************************/
#include "pic24ps2key-keymap.h"

/* indexes into the keymap array */
enum keymapTypes { UNSHIFTED=0, SHIFTED, CAPSLOCKED };

/* Keyboard codes used by the driver software only */
#define LEFTSHIFT  0x12
#define RIGHTSHIFT 0x59
#define CAPSLOCK   0x58

#define KEY_RELEASE_CODE  0xF0
#define EXTENDED_KEY_CODE 0xE0


/**** VARIABLES ************************************************************/
static volatile unsigned char nextBuffPos; // points to next position to write to
static volatile unsigned char readBuffPos; // points to next position to read from
static volatile unsigned char intBitCount; // Counts the bits in a data byte ( 11 )
static volatile unsigned short intWord; // holds incoming bits
static volatile unsigned char keyBuffer[16]; // holds a buffer of keys pressed

/**** FORWARD DECLARATIONS *************************************************/



/**** CODE *****************************************************************/


/****************************************************************************
*
*   Function:   PIC24ps2keyInit
*   inputs:     none
*   returns:    none
*   Purpose:    Setup the I/O used by the keyboard system.
*
****************************************************************************/
void PIC24ps2keyInit(void)
{
    // Setup the interrupt routines variables
    intWord = 0;
    intBitCount = 0;
    nextBuffPos = 0;
    readBuffPos = 0;
    memset((void *)keyBuffer, 0, 16);
    
    // Set this to be a low priority interrupt, so another interrupt can come in
    // TBD - see how it works first.
    
    // Setup the INT0 pin as an interrupt input
    // No action needed - pin is set to input on reset.
    
    // Setup the Data pin as an input.
    // No action needed - pin is set to input on reset.
    
    // Setup the INT0 pin interrupt direction - fire on negative edge
    INTCON2bits.INT0EP = 1;
    
    // Enable the interrupt routine
    IFS0bits.INT0IF = 0;  // Clear the flag, in case of spurious sets
    IEC0bits.INT0IE = 1;  // Enable interrupt
}


/****************************************************************************
*
*   Function:   PIC24ps2keyGetByte
*   inputs:     none
*   returns:    a byte from the keyboard buffer
*   Purpose:    Low level interface to the keyboard driver.
*               This routine will block until data bytes come in.
*
****************************************************************************/

unsigned char PIC24ps2keyGetByte( void )
{
    unsigned char val;
    
    while (nextBuffPos == readBuffPos)
        ; /* wait for a byte to appear */

    val = keyBuffer[readBuffPos++];
    readBuffPos = readBuffPos & 0x0F; // rotate round the 16 byte buffer
    
    return val;
}



/****************************************************************************
*
*   Function:   PIC24ps2keyGetKey
*   inputs:     none
*   returns:    A character from the keyboard.
*   Purpose:    When called, will parse characters in the 
*               Your application reads this to get 2 byte pairs for a key press.
*               This function will block until a key is pressed.
*
****************************************************************************/

unsigned int PIC24ps2keyGetKey( void )
{
    unsigned char val;
    unsigned char nokey = 1;
    static unsigned char keyset = UNSHIFTED;
    
    
    do { 
        // Wait for a byte form the keyboard
        val = PIC24ps2keyGetByte();
        
        // Is this byte the first of a two byte 'key up' message?
        if ( val == KEY_RELEASE_CODE ) {
            
            // yes - so get the byte indicating which key went up
            val = PIC24ps2keyGetByte();    
        
            if ( (val == RIGHTSHIFT) || (val == LEFTSHIFT) )
                keyset = UNSHIFTED;
        
        } else if ( val == EXTENDED_KEY_CODE ) {
            // The byte indicates that a key has been pressed, but there are two bytes.
            // These are special extended keys.
            val = PIC24ps2keyGetByte();
            // At this point, you could look for the 'extended keys' that return two bytes
            // rather than just ignore them
        } else if ( val < 0x85 ) {
            // The byte is the code for a key being pressed
            // Some keys, like shift, get handled in here.
            // Normal keys will be returned to the user
            if (( val == LEFTSHIFT ) || ( val == RIGHTSHIFT )) {
                keyset = SHIFTED;
            } else if (val == CAPSLOCK ) {
            
                if ( keyset == CAPSLOCKED )
                    keyset = UNSHIFTED;
                else
                    keyset = CAPSLOCKED;
                // You might want to try turning the capslock LED on & off here
            } else 
                nokey = 0; /* Got a valid key - break out of loop to return it */
        }
            
    } while ( nokey );
    
    return keymap[val][keyset];
}


/****************************************************************************
*
*   Function:   _INT0Interrupt
*   inputs:     none, its an interrupt
*   returns:    none, its an interrupt
*   Purpose:    The interrupt for the INT0 pin - triggered when the pin goes
*               low. A simple state machine tracks the bits for a complete
*               sequence of clocks from the keyboard to record and store a
*               series of bytes coming in.
*               The bit sequence includes a parity bit, but ignore this.
*
****************************************************************************/

void __attribute__((interrupt, auto_psv)) _INT0Interrupt(void)
{
    
    IFS0bits.INT0IF = 0;  // Clear flag - otherwise we come straight back here

    intWord = intWord >> 1;
    
    // Test level of data pin, and store the bit away.
    if ( PORTBbits.RB10 ) 
        intWord |= 0x400;
        
    ++intBitCount;
    
    if ( intBitCount == 11 ) {
        keyBuffer[nextBuffPos++] = ( intWord >> 1 ) & 0xFF;
        nextBuffPos = nextBuffPos & 0x0F; // rotate round 16 byte buffer
        intBitCount = 0;
        intWord = 0; 
    }
}


