
;*****************************************************************************        
;
;   Module:     wakeup.asm
;               
;   Author:     Mike Hibbett, mike.hibbett@gmail.com
;                                                                  
;   Version:    1.0 9/5/06                                                
;
;               Demonstrates use of the 'wakeup on change' interrupt
;
;   This example is based on the PIC16F688, running at 4MHz.
;   The speed of the processor only affects the keyboard debounce delay
;   routine delay100us.
;
;   The keyboard is a 12 button keypad, arranged in a 3 x 4 layout
;   The three column wires connect to inputs on pins RA0, RA1 and RA3.
;   Pullup resistors (eg, 4K7) should be fitted to those lines
;   The four row signals go to pins RC0,RC1,RC2 and RC3, which are configured
;   as outputs. No other I/O pins are used.
;
;   The PIC16F688 has an 'interrupt on change' feature on the PORTA port. 
;   Each individual pin may be have the interrupt feature enabled or disabled.
;
;   The key pressed is returned in the keyPressed variable. Values returned 
;   will be:
;
;   Key Id      Byte Returned
;   0           0x1A
;   1           0x19
;   2           0x13
;   3           0x2A
;   4           0x29
;   5           0x23
;   6           0x4A
;   7           0x49
;   8           0x43
;   9           0x8A
;   10          0x89
;   11          0x83
;
;*****************************************************************************        

    list p=16F688, st=OFF, x=OFF, n=0

    
    errorlevel -302
    errorlevel -306

    #include <p16F688.inc>

   

;*****************************************************************************        
;
;   Function :  config register setting
;               
;
;   Input:      N/A
;
;   Output:     N/A
;
;*****************************************************************************        
    ; __config  0b0000 0000 0000 0000
    ;             0000                  not implemented
    ;                  0                Clock monitor disabled
    ;                   0               Switch Over disabled
    ;                    00             Brown out disabled
    ;                       1           No data code protection
    ;                        1          No code protection
    ;                         0         MCLR internal
    ;                          0        Power up timer enabled
    ;                            0      Watchdog disabled
    ;                             001   XT Clock ( 4MHz )
    
    __config    0x0C1



;*****************************************************************************        
;
;               Variable definitions
;               We get 0x20 to 0x7F, A0 - FF, 120 - 17F in 3 banks
;               The last 16 bytes are echoed in each bank.
;
;   Input:      N/A
;
;   Output:     N/A
;
;*****************************************************************************        

delayA              EQU 0x20        ; Used bu 100us delay routine
delayB              EQU 0x21        
delayC              EQU 0x22        
flags1              EQU 0x23        ; interrupt flags

keyPressed          EQU 0x24        ; Stores actual key pressed
keyPressed2         EQU 0x25        ; temporary variable, used to debouce key
pinBuffer           EQU 0x26        ; temporary variable

PCLATH_TEMP         EQU 0x7d        ; Exists in all banks
STATUS_TEMP         EQU 0x7e        ; Exists in all banks
W_TEMP              EQU 0x7f        ; Exists in all banks



;*****************************************************************************        
;
;               Constants definitions
;               Handy, descriptive names for constants. 
;
;   Input:      N/A
;
;   Output:     N/A
;
;*****************************************************************************        

INT_KEYPRESS_FLAGS1     EQU 0       ; Bit in flags1. Indicates key pressed



;*****************************************************************************        
;
;   Function :  Reset vector
;               Hardware entry point to the code
;
;   Input:      None.
;
;   Output:     N/A
;
;*****************************************************************************        
        
    ORG     0    
    GOTO    main



;*****************************************************************************        
;
;   Function :  Interrupt vector
;               Hardware entry point for interrupts
;               This vector handles all external and internal interrupts
;
;   Input:      None.
;
;   Output:     N/A
;
;*****************************************************************************        
    
    ORG    4
    
INTERRUPT
    ; Save the STATUS and PCLATH registers
    movwf   W_TEMP
    swapf   STATUS, W
    clrf    STATUS
    movwf   STATUS_TEMP    
    movfw   PCLATH
    movwf   PCLATH_TEMP
    clrf    PCLATH
    
    ; Test to see if the interrupt was as a result of a key being pressed
    btfss   INTCON, RAIF
    goto    testOtherInts
    
    ; Clear the mismatch which caused the interrupt
    movfw   PORTA                   
    
    bcf     INTCON, RAIF
    bsf     flags1, INT_KEYPRESS_FLAGS1
        
testOtherInts
    ; Check for other interrupt sources...        
        
intDone        
    ; Restore the STATUS and PCLATH registers
    movfw   PCLATH_TEMP
    movwf   PCLATH
    swapf   STATUS_TEMP, W
    movwf   STATUS
    swapf   W_TEMP,F
    swapf   W_TEMP, W
    retfie
    


;*****************************************************************************        
;
;   Function :  main
;               Location branched to by reset vector
;
;   Input:      N/A
;
;   Output:     Never returns
;
;*****************************************************************************        

main
    bcf     STATUS, RP0

    ; clear all user ram, 0x20 - 0x7F
    movlw   0x20
    movwf   FSR
    
m000    
    clrf    INDF
    incf    FSR,F
    movlw   0x80
    subwf   FSR,W
    btfss   STATUS,Z
    goto    m000
    
    ; Setup the PIC's peripherals
    call    HWInit
    
    ; Wait 1s for clean powerup
    movlw   D'100'
    call    delayLong

    ; Enable keyboard interrupts
    bsf     INTCON, RAIE
    bsf     INTCON, GIE         ; Enable global interrupts
    
    ; Enabled keyboard detection
    bsf     STATUS,RP0              
    movlw   0x0B
    movwf   IOCA
    bcf     STATUS,RP0                  
    
    ; Main Loop
    ; cycle round looking for keypresses or incoming RS232 comms
    
lp    
    ; Turn of oscillator and enter ultra low power mode
    sleep
    
    ; Did we wake up due to a key press?
    btfss   flags1, INT_KEYPRESS_FLAGS1
    goto    lp                  ; - No, so go back to sleep
    
    ; Yes, wakeup caused by keypress
    
    ; disable any further key press detection until we 
    ; have finished testing the keys
    bsf     STATUS,RP0              
    movlw   0x00
    movwf   IOCA
    bcf     STATUS,RP0              
        
    ; Clear the interrupt flag
    bcf     flags1, INT_KEYPRESS_FLAGS1
    
 
    call    getKey
    
    
    ; DO YOUR KEY PROCESSING HERE
    

    ; Important. Read the port again, to clear any false 'wake up' conditions
    movfw   PORTA                   
    
    ; Enabled keyboard detection
    bsf     STATUS,RP0              
    movlw   0x0B
    movwf   IOCA
    bcf     STATUS,RP0   

    goto    lp
        
    
    
;*****************************************************************************        
;
;   Function :  getKey
;               Scans the keyboard and returns a key code, zero if no
;               key found
;
;   Input:      N/A
;
;   Output:     kye code in keyPressed
;
;   Note:       The routine will set each PORTC bit high in turn, and
;               look to see if the lines on PORTA change.
;               If they do, set at bit in the high nibble of keyPressed
;               The low nibble of keypressed will hold the 3 bits of the
;               PORTA pins. Together, this will produce a unique code
;               for the given key pressed (see top of file).
;
;*****************************************************************************        
getKey
    ; Get the key detection line data
    movfw   PORTA
    andlw   0x0B
    movwf   keyPressed
    
    clrf    pinBuffer
    
    bsf     pinBuffer, 0
    movfw   pinBuffer
    movwf   PORTC
    nop
    nop
    movfw   PORTA
    andlw   0x0B
    subwf   keyPressed, W
    btfsc   STATUS, Z
    goto    hk001               ; Not that key
    
    bsf     keyPressed, 4   
    bcf     pinBuffer, 0
    movfw   pinBuffer
    movwf   PORTC
    goto    hkchk               

hk001
    bsf     pinBuffer, 1
    movfw   pinBuffer
    movwf   PORTC
    nop
    nop
    movfw   PORTA
    andlw   0x0B
    subwf   keyPressed, W
    btfsc   STATUS, Z
    goto    hk002               ; Not that key
    
    bsf     keyPressed, 5   
    bcf     pinBuffer, 1
    movfw   pinBuffer
    movwf   PORTC
    goto    hkchk               

hk002
    bsf     pinBuffer, 2
    movfw   pinBuffer
    movwf   PORTC
    nop
    nop
    movfw   PORTA
    andlw   0x0B
    subwf   keyPressed, W
    btfsc   STATUS, Z
    goto    hk003               ; Not that key
    
    bsf     keyPressed, 6   
    bcf     pinBuffer, 2
    movfw   pinBuffer
    movwf   PORTC
    goto    hkchk               

hk003
    bsf     pinBuffer, 3
    movfw   pinBuffer
    movwf   PORTC
    nop
    nop
    movfw   PORTA
    andlw   0x0B
    subwf   keyPressed, W
    btfsc   STATUS, Z
    goto    hk004               ; Not that key
    
    bsf     keyPressed, 7   
    bcf     pinBuffer, 3
    movfw   pinBuffer
    movwf   PORTC
    goto    hkchk               

hk004
    ; No key found - must have been transient. So just exit
    clrf    keyPressed
    clrf    PORTC
    goto    hkexit

hkchk
    ; 40ms delay
	movlw	D'4'		
	call    delayLong
    
    ; Debounce

    movfw   PORTA
    andlw   0x0B
    movwf   keyPressed2
    
    clrf    pinBuffer
    
    bsf     pinBuffer, 0
    movfw   pinBuffer
    movwf   PORTC
    nop
    nop
    movfw   PORTA
    andlw   0x0B
    subwf   keyPressed2, W
    btfsc   STATUS, Z
    goto    hk1001               ; Not that key
    
    bsf     keyPressed2, 4   
    bcf     pinBuffer, 0
    movfw   pinBuffer
    movwf   PORTC
    goto    hk1004               

hk1001
    bsf     pinBuffer, 1
    movfw   pinBuffer
    movwf   PORTC
    nop
    nop
    movfw   PORTA
    andlw   0x0B
    subwf   keyPressed2, W
    btfsc   STATUS, Z
    goto    hk1002               ; Not that key
    
    bsf     keyPressed2, 5   
    bcf     pinBuffer, 1
    movfw   pinBuffer
    movwf   PORTC
    goto    hk1004               

hk1002
    bsf     pinBuffer, 2
    movfw   pinBuffer
    movwf   PORTC
    nop
    nop
    movfw   PORTA
    andlw   0x0B
    subwf   keyPressed2, W
    btfsc   STATUS, Z
    goto    hk1003               ; Not that key
    
    bsf     keyPressed2, 6   
    bcf     pinBuffer, 2
    movfw   pinBuffer
    movwf   PORTC
    goto    hk1004               

hk1003
    bsf     pinBuffer, 3
    movfw   pinBuffer
    movwf   PORTC
    nop
    nop
    movfw   PORTA
    andlw   0x0B
    subwf   keyPressed2, W
    btfsc   STATUS, Z
    goto    hk1004               ; Not that key
    
    bsf     keyPressed2, 7   
                
hk1004
    clrf   PORTC
    
    ; Still same value?
    movfw   keyPressed2
    subwf   keyPressed, W
    btfss   STATUS, Z
    clrf    keyPressed    
    
    ; Now, wait for key to be released
hklp
    movfw   PORTA
    andlw   0x0B
    sublw   0x0B
    btfss   STATUS, Z
    goto    hklp    
    
    ; If it was a false key press, ignore
    movfw   keyPressed
    btfsc   STATUS, Z
    goto    hkexit     
    
hkexit2
    ; 40ms delay, to ignore false keypresses
	movlw	D'4'		
	call    delayLong        

hkexit    
    return
        
  

;*****************************************************************************        
;
;   Function :  HWInit
;               Sets up the hardware peripherals
;
;   Input:      N/A
;
;   Output:     N/A
;
;   Note:       
;
;*****************************************************************************        
HWInit
    clrf    PORTA
    clrf    PORTC
    
    ; Make all digital I/O ports available
    movlw   0x07
    movwf   CMCON0
    bsf     STATUS,RP0             
    clrf    ANSEL
    
    movlw   0x3F
    movwf   TRISA
    movlw   0x30                    ; All outputs except rx/tx
    movwf   TRISC
            
    bcf     STATUS,RP0              ;Restore Bank0    
    return



;*****************************************************************************        
;
;   Function :  delayLong
;            Waits the specified number of 10ms periods
;
;   Input:   # of 10ms periods, in w
;
;   Output:  None
;
;   Note:       Uses delayA variable
;
;*****************************************************************************        
delayLong
	movwf	delayA
	call	delay10ms
    decfsz	delayA, F
    goto	$-2
    return



;*****************************************************************************        
;
;   Function :  delay10ms
;               An approximate 10ms delay
;
;   Input:      N/A
;
;   Output:     N/A
;
;   Note:       Uses delayB variable
;
;*****************************************************************************        
delay10ms
	movlw   D'97'
	movwf   delayB
d10ms
    call	delay100us    
    decfsz	delayB, F
    goto	d10ms
    return
    
    
    
;*****************************************************************************        
;
;   Function :  delay100us
;               delay exactly 100us at 4MHz clock
;
;   Input:      N/A
;
;   Output:     N/A
;
;   Note:       Uses delayC variable
;
;*****************************************************************************        
                        ; Timings
delay100us              ; 2 (for the call)
    movlw   0x13        ; 1  (count of 19)
    movwf   delayC      ; 1
dlyus
    nop             
    nop                 ; 2
    
    decfsz  delayC, F   ; 1/2
    goto    dlyus       ; 2    
    return              ; 2    
    
    
    END                ; End of program
