;*****************************************************************************        
;
;   Module:     bootload.inc
;               
;   Author:     Mike Hibbett, mikehibbett@oceanfree.net 
;                                                                  
;   Version:    1.0 11/10/05                                              
;
;               The bootloader code.
;               It contains *all* the code required to initialise the hardware,
;               interact with a host over RS232 and reprogram the flash.
;
;               You should not need to change this, unless your hardware has
;               some special start-up requirements ( like some pins must be
;               turned into output immediately after reset. Eg, to turn of a
;               motor, etc. ) If you must, place that code by the comment
;                 ; Hardware Initial Setup
;
;
;*****************************************************************************        



;*****************************************************************************        
;
;   Function :  bootloader

;               This routine is always in low memory immediately after the
;               reset and interrupt vectors (along with the remapped vectors.)
;               It provides initial hardware setup, detection of bootloader 
;               activation request. It must never exceed 0x600-0x1C hex bytes
;               in size.
;
;   Input:      None.
;
;   Output:     Returns if bootloader activation is not required, resulting
;               in the main application starting.
;
;*****************************************************************************        
Bootloader

    ; Hardware Initial Setup
    ; Place any addition initial hardware setup here, if required.


    ;   First, delay for a little to allow power to settle
    movlw   D'10'               ; 100ms
    call    UIWait10ms  

    
    ; do initial bootloader hardware setup ( port directions and levels )
    
    movlw   0x07    ; Disconnect comparitor from I/O pins
    movwf   CMCON 
    
    movlw   0x00    ; reference voltage not used
    movwf   CVRCON 
    
    movlw   0x0F    ; Disconnect A/D from I/O pins
    movwf   ADCON1 
    

    ; Setup the RS232 Interface Pins.
    ; This just means setting the transmit pin to output, high level.

    bcf     TRIS_TX_PIN, TX_PIN_BIT     ; Make it output
    bsf     PORT_TX_PIN, TX_PIN_BIT     ; Make it high level
          
    ; OK, execute the RS232 Download routine.
    ; We never return from this code.
    
    ; Sync with the PC by exchanging 'X' bytes, followed by a G for Go!
    movlw   RX_RETRIES          ; Count failed attempts to communicate
    movwf   rxRetries
BL002
    movlw   0x01
    call    UIWait10ms
    movlw   'X'
    call    RS232TxByte
    call    RS232RxByteNB
    
    ; Did we get the correct response?
    movlw   'X'
    subwf   hostByte, W
    btfsc   STATUS, Z
    goto    BL003               ; Yes! Enter bootloader
    
    ; No. Was it a timeout event in RS232RxByteNB
    movlw   0x00
    subwf   hostByte, W
    btfss   STATUS, Z
    bra     BL002               ; No, so go back and look for another character
    
    ; Yes. Decrement retry counter. If it reaches zero, return to the application
    dcfsnz  rxRetries, F
    return        
    bra     BL002

BL003       
    movlw   0x01
    call    UIWait10ms
    movlw   'G'
    call    RS232TxByte

    ; Then wait for a flash programming packet. This is 
    ; AAD.....DC
    ;  AA == 16 bit address to write to 
    ;  D..D 64 data bytes
    ;  C  check byte
    ; PIC returns Y for ok, C for checkdigit error, F for flash error
    ;  Z for exiting ok
    ; Send address of 0x0000 ( and no data or check byte ) to end transfer
    ; Only addresses above bootloader are acceptable

BLProg
    call    RxFlashBlock
    
    ; Examine result variable following call to RxFlashBlock
    ; Termination packet received?
    movlw   'Z'
    subwf   progRxResult, W
    btfsc   STATUS, Z
    bra     BLEnd
    
    ; Error in received packet?
    movlw   'C'
    subwf   progRxResult, W
    btfsc   STATUS, Z
    bra     BLError
    
    call    ProgramFlashBlock
    
    ; Error during programming?
    movlw   'F'
    subwf   progRxResult, W
    btfsc   STATUS, Z
    bra     BLError2

    ; Echo OK to Host
    movlw   0x01
    call    UIWait10ms
    movlw   'Y'
    call    RS232TxByte

    bra     BLProg
    
BLEnd
    ; Echo exiting to Host
    movlw   0x01
    call    UIWait10ms
    movlw   'Z'
    call    RS232TxByte

    ; Wait a bit, then jump to application
    movlw   D'100'
    call    UIWait10ms
    return

BLError
    ; Echo Error to Host
    movlw   0x01
    call    UIWait10ms
    movlw   'C'
    call    RS232TxByte
    bra     BLProg

BLError2
    ; Echo Error to Host
    movlw   0x01
    call    UIWait10ms
    movlw   'F'
    call    RS232TxByte
    bra     BLProg
    
    
    
;*****************************************************************************        
;
;   Function : UIWait10ms
;              delays for a multiple of 10ms
;
;   Input:     multiple in W
;
;*****************************************************************************        
UIWait10ms
    movwf   delay3 
    
d1ms001
    movlw   D'200'
    call    UIWait50us
    decfsz  delay3, F 
    bra     d1ms001
    return



;*****************************************************************************        
;
;   Function :  waitBitTime
;               Waits for 1 bit time, less a constant time of 8 cycles
;               because the caller loop overhead is 8 cycles
;               Baud rate is 115200 @ 40MHZ 1 bit time = 8.68us, say 87 cycles
;               Therefore we want, including call/return overhead, 79 cycles
;
;   Input:      None
;
;   Output:     None
;  
;
;*****************************************************************************        
waitBitTime
    movlw   BIT_LOOP_VAL
    movwf   bitDelay        

wbt001    
    decfsz  bitDelay,F      
    bra    wbt001          

    IF (BIT_EXTRA_CLOCKS == 1) 
    nop
    ENDIF

    IF (BIT_EXTRA_CLOCKS == 2) 
    nop
    nop
    ENDIF
    
    nop    
    return


;*****************************************************************************        
;
;   Function :  waitBitHalfTime
;               Waits for 1.5 bit time
;               Baud rate is 115200. 1.5 bit time = 13us, 130 cycles @ 40MHZ
;
;   Input:      None
;
;   Output:     None
;  
;
;*****************************************************************************        
waitBitHalfTime
    movlw   BITHALF_LOOP_VAL
    movwf   bitDelay        

wbht001    
    decfsz  bitDelay,F      
    bra    wbht001           

    IF (BITHALF_EXTRA_CLOCKS == 1) 
    nop
    ENDIF

    IF (BITHALF_EXTRA_CLOCKS == 2) 
    nop
    nop
    ENDIF
    
    nop    
    return






;*****************************************************************************        
;
;   Function : UIWait50us
;              delays for a multiple of 50us
;
;   Input:     multiple in W
;                         
;
;*****************************************************************************        
UIWait50us              
    movwf   delay2       

d1us002
    movlw   DELAY_50US_LOOP_VAL           
    movwf   delay1                
    
d1us001      
    decfsz  delay1, F         
    bra     d1us001                   
    
    decfsz  delay2, F   
    bra     d1us002     
    return                    



;*****************************************************************************        
;
;   Function :  RS232TxByte
;               Sends byte in w over the rs232 interface
;               Baud rate is 115200
;   Input:      byte in W
;
;   Output:     None
;
;*****************************************************************************        
RS232TxByte
    movwf   hostByte 
    
    movlw   0x08
    movwf   bitCount 
    
    ; start bit
    bcf     PORT_TX_PIN, TX_PIN_BIT 

rt001    
    call    waitBitTime             ; wait 8.7us - 8 cycles
    
    rrcf    hostByte,F 
    btfss   STATUS, C 
    bcf     PORT_TX_PIN, TX_PIN_BIT 
    btfsc   STATUS,C 
    bsf     PORT_TX_PIN, TX_PIN_BIT 
    decfsz  bitCount,F 
    bra     rt001
    
    call    waitBitTime             ; wait 8.7us - 8 cycles
    nop
    nop
    nop
    nop
    nop
    nop
    
    ; stop bit
    bsf     PORT_TX_PIN, TX_PIN_BIT
    call    waitBitTime             ; wait 8.7us - 8 cycles
    nop
    nop
    nop
    nop
    nop
    nop    
    return
    


;*****************************************************************************        
;
;   Function :  RS232RxByteNB
;               Receives a byte over the rs232 interface
;               Baud rate is 115200
;               Returns a 0x00 if it times out.
;   Input:      None
;
;   Output:     byte in hostByte
;
;               NOTE: This is a Non Blocking call
;
;*****************************************************************************        
RS232RxByteNB
    movlw   0x08
    movwf   bitCount 
    clrf    hostByte 
    
    movlw   ( ( RX_BYTE_TIMEOUT >> 8 ) & 0xFF ) 
    movwf   rxTimeoutH
    movlw   ( RX_BYTE_TIMEOUT & 0xFF )
    movwf   rxTimeoutL
        
renb002
    ; Have we timed out yet?
    
    movlw   0x01
    subwf   rxTimeoutL, F
    movlw   0x00
    subwfb  rxTimeoutH, F
    btfss   STATUS, C
    return

renb001
    btfsc   PORT_RX_PIN, RX_PIN_BIT 
    bra     renb002
   
    call    waitBitHalfTime     ; Past the start bit to middle of first bit
   
renb003
    bsf     STATUS,C 
    btfss   PORT_RX_PIN, RX_PIN_BIT 
    bcf     STATUS,C 
    rrcf    hostByte, F 
    call    waitBitTime       ; wait 1 bit time - 8 cycles
    decfsz  bitCount,F 
    bra     renb003        
    return
;*****************************************************************************        
;
;   Function :  RS232RxByte
;               Receives a byte over the rs232 interface
;               Baud rate is 115200
;   Input:      None
;
;   Output:     byte in hostByte
;
;               NOTE: This is a blocking call
;
;*****************************************************************************        
RS232RxByte
    movlw   0x08
    movwf   bitCount 
    clrf    hostByte 
    
re002
    btfsc   PORT_RX_PIN, RX_PIN_BIT 
    bra     re002
   
    call    waitBitHalfTime     ; Past the start bit to middle of first bit
   
re003
    bsf     STATUS,C 
    btfss   PORT_RX_PIN, RX_PIN_BIT 
    bcf     STATUS,C 
    rrcf    hostByte, F 
    call    waitBitTime       ; wait 1 bit time - 8 cycles
    decfsz  bitCount,F 
    bra     re003        
    return
 


;*****************************************************************************        
;
;   Function :  RxFlashBlock
;               Receives a 64 byte data block over RS232
;               Baud rate is 115200
;
;   Input:      None. 
;
;   Output:     progAddHigh, progAddLow, progRxResult, prog buffer modified
;
;               NOTE: Returns result in progRxResult:
;               'Y' Block received ok
;               'C' Check byte error
;               'Z' Termination requested
;
;               Received data is not echoed ( a result byte is echoed later)
;
;*****************************************************************************        
RxFlashBlock
    call    RS232RxByte
    movf    hostByte, W
    movwf   progRxCheck     ; Init Check digit
    movwf   progAddHigh   
    call    RS232RxByte
    movf    hostByte, W
    iorwf   progRxCheck,W   ; Zero address sent?
    btfsc   STATUS,Z
    bra     RXF001          ; Yes - exit now, no more data is sent
    movf    hostByte, W
    movwf   progAddLow
    addwf   progRxCheck, F
    
    ; setup pointer to where received data will go
    movlw   0x40
    movwf   FSR0L
    clrf    FSR0H

RXF000
    call    RS232RxByte
    movf    hostByte, W
    movwf   POSTINC0
    addwf   progRxCheck, F
    btfss   FSR0L, 7
    bra     RXF000
    
    ; All bytes received. Now get Check byte, and compare with ours
    call    RS232RxByte
    movf    hostByte, W
    subwf   progRxCheck, W
    btfss   STATUS, Z
    bra     RXF002          ; CheckSum Error 

    ; No errors    
    movlw   'Y'
    movwf   progRxResult
    return
    
    ; Checksum Error
RXF002
    movlw   'C'
    movwf   progRxResult
    return
           
    
RXF001
    movlw   'Z'
    movwf   progRxResult
    return



;*****************************************************************************        
;
;   Function :  ProgramFlashBlock
;               Programs a 64 byte data block into the flash
;               
;   Input:      None
;
;   Output:     progAddHigh, progAddLow, progRxResult, prog buffer modified
;
;               NOTE: Returns result in progRxResult:
;               'Y' Programmed OK
;               'F' Flash programming error
;
;               A result byte is echoed to the host later on.
;
;*****************************************************************************        
ProgramFlashBlock

    ; First, erase the block requested
    clrf    TBLPTRU
    movf    progAddHigh, W
    movwf   TBLPTRH
    movf    progAddLow, W
    movwf   TBLPTRL
    
    bsf     EECON1, EEPGD
    bcf     EECON1, CFGS
    bsf     EECON1, WREN
    bsf     EECON1, FREE
    bcf     INTCON, GIE
    
    movlw   0x55
    movwf   EECON2
    movlw   0xAA
    movwf   EECON2
    bsf     EECON1, WR              ; Start erase; CPU stalls until done
    
    ; Now program the erased block. We do it in 2, 32byte writes
    
    tblrd*-                         ; Dummy read decrement
    
    movlw   0x40
    movwf   FSR0L
    clrf    FSR0H
    
    movlw   D'32'
    movwf   progCounter

PFB001
    movf    POSTINC0, W
    movwf   TABLAT
    tblwt+*                         ; inc address then write
    decfsz  progCounter, F
    bra     PFB001
    
    bsf     EECON1, EEPGD
    bcf     EECON1, CFGS
    bsf     EECON1, WREN

    movlw   0x55
    movwf   EECON2
    movlw   0xAA
    movwf   EECON2
    bsf     EECON1, WR              ; Start erase; CPU stalls until done
    bcf     EECON1, WREN
    
    ; Now do next 32 bytes
    movlw   0x60
    movwf   FSR0L
    clrf    FSR0H
    
    movlw   D'32'
    movwf   progCounter

PFB002
    movf    POSTINC0, W
    movwf   TABLAT
    tblwt+*                         ; inc address then write
    decfsz  progCounter, F
    bra     PFB002
    
    bsf     EECON1, EEPGD
    bcf     EECON1, CFGS
    bsf     EECON1, WREN

    movlw   0x55
    movwf   EECON2
    movlw   0xAA
    movwf   EECON2
    bsf     EECON1, WR              ; Start erase; CPU stalls until done
    bcf     EECON1, WREN
    
    movlw   'Y'
    movwf   progRxResult
    
    return
