;*****************************************************************************        
;
;   Module:     cw2app.inc
;               
;   Author:     Mike Hibbett 
;                                                                  
;   Version:    0.1 10/06/05                                                  
;
;               The main application code of the CamerWatch2 Project, entered
;               following execution of the bootloader. 
;
;
;*****************************************************************************        



;*****************************************************************************        
;
;   Macro :  DISPLAY
;            Helper macro to write a string to the display
;
;   Input:   Address of String to display
;
;   Output:  W modified. UIDspStr called. User informed :o)
;
;*****************************************************************************        
DISPLAY     macro   addr
    movlw   HIGH    addr
    movwf   TBLPTRH
    movlw   LOW    addr
    movwf   TBLPTRL
    call    UIDspStr
            endm


;*****************************************************************************        
;
;   Function :  Main
;               This is the entry point of the actual application program
;
;   Input:      None.
;
;   Output:     None, never returns
;
;*****************************************************************************        
Main
    call    AppHWInit
    
    ; Advise the user that the application has started
    
    ; Check the eeprom - initialise it has never been initialised
    call    EEPCheck
    
    call    InitDspBarSet

    call    DspHome
    call    DspClear

    ; In most cases this display is quickly replaced 
    DISPLAY STRAcqGPSL1
    call    DspLine2    
    DISPLAY STRAcqGPSL2
    
    ; Get number of records in the eeprom and other constants
    call    GetSettings
    
    movlw   ALERT_POWER_UP
    movwf   tuneCurrent
            
Mainloop
    ; If we go from out of lock to in lock, or the other way round,
    ; then give a little beep
    movf    flags1, W
    xorwf   flags2, W
    andlw   0x02    ; Extract just the difference between the lock bits
    btfsc   STATUS,Z
    goto    ml0002
    
    ; Lock has changed. Update application copy, and sound alert 
    btg     flags1, GPS_LOCK_FLAG1

    movlw   ALERT_GPS_LOCK_OK
    btfss   flags1, GPS_LOCK_FLAG1
    movlw   ALERT_GPS_LOCK_LOST
    movwf   tuneCurrent      
    

ml0002
    ; If we are currently out of lock, flag that the speed data is invalid 
    btfss   flags1, GPS_LOCK_FLAG1  
    bcf     flags1, SPEEDHEAD_VALID_FLAG1

    ; if alt/speed data available, copy across and clear the flag
    btfss   flags2, NEWSPEEDDATAFLAG2
    goto    ml000a
    
    ; Tell the foreground task that the speed/heading data is valid
    bsf     flags1, SPEEDHEAD_VALID_FLAG1
    
    bcf     INTCON, GIE            
    movf    iHead1, W, BANKED           
    movwf   Head1, BANKED           
    movf    iHead2, W, BANKED           
    movwf   Head2, BANKED           
    movf    iHead3, W, BANKED           
    movwf   Head3, BANKED           
    movf    iSpeedH, W
    movwf   SpeedH
    movf    iSpeedL, W
    movwf   SpeedL
    bcf     flags2, NEWSPEEDDATAFLAG2   
    bsf     INTCON, GIE             

ml000a
    btfss   flags2, NEWALTDATAFLAG2
    goto    ml000p
    
    bcf     INTCON, GIE  
    movf    iAlt1, W
    movwf   Alt1
    movf    iAlt2, W
    movwf   Alt2
    movf    iAlt3, W
    movwf   Alt3
    movf    iAlt4, W
    movwf   Alt4
    bcf     flags2, NEWALTDATAFLAG2
    bsf     INTCON, GIE             
    
ml000p
    ; have we received new position data?
    btfss   flags2, NEWDATAFLAG2
    goto    ml001
       
    bcf     INTCON, GIE             ; Disable all interrupts
    bcf     flags2, NEWDATAFLAG2
    
    ; copy the data out of the interrupt area
    movf    iLatDeg, W
    movwf   LatDeg
    movf    iLongDeg, W
    movwf   LongDeg
    movf    iLatmmhh, W
    movwf   Latmmhh
    movf    iLatmmhh + 1, W
    movwf   Latmmhh + 1
    movf    iLatmmhh + 2, W
    movwf   Latmmhh + 2
    movf    iLongmmhh, W
    movwf   Longmmhh
    movf    iLongmmhh + 1, W
    movwf   Longmmhh + 1
    movf    iLongmmhh + 2, W
    movwf   Longmmhh + 2

    bcf     flags2,NEWDATAFLAG2

    bsf     INTCON, GIE             ; Enable global interrupts

    ; Scan the list of stored camera points

    ; First, calculate the cosine of the latitude angle
    call    calcCosineH
    movwf   cosineValueH
    call    calcCosineL
    movwf   cosineValueL    
 
    clrf    currentPosH
    clrf    currentPosL

    ; Set a maximum distance. Any recorded point should be closer than this
    movlw   0x7f
    movwf   scanDist
    movlw   0xff
    movwf   scanDist+1
    movwf   scanDist+2

    ; If we have no camera points, no need to scan list
    movf    numEntriesH, W
    iorwf   numEntriesL, W
    btfsc   STATUS, Z
    goto    ml001       
    
    ; ScanNum will be our pointer to a camera location in eeprom
    movf    numEntriesH, W
    movwf   scanNumH
    movf    numEntriesL, W
    movwf   scanNumL


    ; The camera location scanning loop is in the include file
    #include "scan.inc"

;               At this point, we have the following information
;               1) CurrentPos - gives eeprom index of closest camera. zero if none
;               2) ScanDist - the distance-squared to the camera. 
;                  distance is in 1/100 of a minute of arc


ml001    
    ; If the camera has changed since the last loop, clear the alert
    ; mask bit.
    movf    currentPosH, W
    subwf   lastCurrentPosH, W
    btfss   STATUS, Z
    bra     ml00113
    
    movf    currentPosL, W
    subwf   lastCurrentPosL, W
    btfsc   STATUS, Z
    bra     ml00114
    
ml00113    
    bcf     flags1, MASK_ALERT_FLAG1
    movf    currentPosH, W
    movwf   lastCurrentPosH
    movf    currentPosL, W
    movwf   lastCurrentPosL

ml00114
    ; First check the record and erase keys. But these only work when we are in lock,
    ; so test that flag first, and ignore them if not in lock
    btfss   flags2, INLOCKFLAG2
    bra     ml00111

    btfsc   flags2, KEY_CLEAR_FLAG2
    call    ClearKeyPressed
    
    btfsc   flags2, KEY_RECORD_FLAG2
    call    RecordKeyPressed

ml00111
    bcf     flags2, KEY_CLEAR_FLAG2
    bcf     flags2, KEY_RECORD_FLAG2

    btfsc   flags2, KEY_SELECT_FLAG2
    call    SelectKeyPressed

    btfsc   flags2, KEY_NEXT_FLAG2
    call    NextKeyPressed
            
; The new software, v1.4, does not have a real default display as such.
;    tstfsz  appTmpDspTmr
;    bra     ml00112
;    
;    ; Reload the real default display if we are currently running
;    ; off a temporary display
;    btfss   flags1, TMPDSP_FLAG1
;    bra     ml00112
;    
;    bcf     flags1, TMPDSP_FLAG1    
;    
;    ; Reload the correct default display    
;    clrf    eepAddH
;    movlw   0x04
;    movwf   eepAddL
;    call    EEPRead
;    movf    eepData, W
;    movwf   defaultDisplay
    
ml00112    
    ; call the appropriate display handler
    movf    defaultDisplay, W
    sublw   NUM_DISPLAYS-1
    btfss   STATUS,C
    clrf    defaultDisplay
    
    call    HandleDisplay
   
    goto    Mainloop
           


;*****************************************************************************        
;
;   Function :  GetSettings
;               Gets the application settings from the eeprom
;
;   Input:      None.
;
;   Output:     None.
;
;*****************************************************************************        
GetSettings
    clrf    eepAddH
    movlw   0x02
    movwf   eepAddL
    call    EEPRead
    movf    eepData, W
    movwf   numEntriesH
    incf    eepAddL, F
    call    EEPRead
    movf    eepData, W
    movwf   numEntriesL
    incf    eepAddL, F
    call    EEPRead
    movf    eepData, W
    movwf   defaultDisplay
    return



;*****************************************************************************        
;
;   Function :  ClearKeyPressed
;               Called when the 'Clear Alert/Location' button is pressed.
;               This is only ever reached if we are in lock
;
;   Input:      None.
;
;   Output:     None.
;
;*****************************************************************************        
ClearKeyPressed
    ; if sounding an alert, cancel the alert and make sure that alert stays 
    ; off until the alert condition clears
    ; 
    tstfsz  tuneCurrent
    bra     CKPClear
    bra     CKPDelete
    
CKPClear        
    movlw   ALERT_SILENCE
    movwf   tuneCurrent
    bsf     flags1, MASK_ALERT_FLAG1
    return
    
    ; else...
CKPDelete
    ; 1) Only delete if we have some entries in the first place
    movf    numEntriesL, W
    iorwf   numEntriesH, W  
    btfsc   STATUS,Z
    return  
    
    ; 1a) If currentPos is zero, we dont actually have a location to delete
    movf    currentPosL, W
    iorwf   currentPosH, W  
    btfsc   STATUS,Z
    return  
    
    ; OK, we are going to delete something. So signal the fact, via a beep
    movlw   ALERT_KEY_PRESS
    movwf   tuneCurrent    
    
    ; 2) If the position to delete is not the last one, copy the last one 
    ; into the position, then goto step 3 else go straight to step 3
    
    movf    currentPosL, W
    subwf   numEntriesL, W
    btfss   STATUS,Z
    goto    CKP000

    movf    currentPosH, W
    subwf   numEntriesH, W
    btfsc   STATUS,Z
    goto    CKP001
        
CKP000
    ; Copy the last position to here
    movlw   0x06
    mulwf   numEntriesH
    movf    PRODL, W    
    movwf   eepAddH 
    movlw   0x02
    movwf   eepAddL
    movlw   0x06
    mulwf   numEntriesL
    movf    PRODL, W
    addwf   eepAddL, F
    movf    PRODH, W
    addwfc  eepAddH, F
    
    call    EEPRead
    movwf   Latmmhh
    incf    eepAddL, F
    call    EEPRead
    movwf   Latmmhh+1
    incf    eepAddL, F
    call    EEPRead
    movwf   Latmmhh+2
    incf    eepAddL,F       
    call    EEPRead
    movwf   Longmmhh
    incf    eepAddL, F
    call    EEPRead
    movwf   Longmmhh+1
    incf    eepAddL, F
    call    EEPRead
    movwf   Longmmhh+2

    movlw   0x06
    mulwf   currentPosH
    movf    PRODL, W    
    movwf   eepAddH 
    movlw   0x02
    movwf   eepAddL
    movlw   0x06
    mulwf   currentPosL
    movf    PRODL, W
    addwf   eepAddL, F
    movf    PRODH, W
    addwfc  eepAddH, F
    
    ; Now store the data. This is
;            binary minutes, signed lat          3
;            binary minutes, signed long         3

    movf    Latmmhh, W
    movwf   eepData
    call    EEPWrite
    movf    Latmmhh+1, W
    movwf   eepData
    call    EEPWrite
    movf    Latmmhh+2, W
    movwf   eepData
    call    EEPWrite
    movf    Longmmhh, W
    movwf   eepData
    call    EEPWrite
    movf    Longmmhh+1, W
    movwf   eepData
    call    EEPWrite
    movf    Longmmhh+2, W
    movwf   eepData
    call    EEPWrite
    
CKP001
    ; 3) decrement numEntries by one, and write numEntries back to eeprom
    movlw   0x01                  
    subwf   numEntriesL,F
    btfss   STATUS,C
    decf    numEntriesH,F       ; sub 1 from number of stored points
    
    movf    numEntriesH, W
    movwf   eepData
    movlw   0x02
    movwf   eepAddL
    clrf    eepAddH
    call    EEPWrite
    movf    numEntriesL, W
    movwf   eepData
    call    EEPWrite            ; store number of stored points
    return



;*****************************************************************************        
;
;   Function :  RecordKeyPressed
;               Called when the 'Record Location' button is pressed.
;               This is only ever reached if we are in lock
;
;   Input:      None.
;
;   Output:     None.
;
;*****************************************************************************        
RecordKeyPressed
    ; do not allow a record if memory full
    movlw   HIGH  MAX_LOCATIONS
    subwf   numEntriesH, W
    btfss   STATUS, Z
    goto    RKP001

    movlw   LOW  MAX_LOCATIONS
    subwf   numEntriesL, W
    btfsc   STATUS, Z
    return      
    
RKP001    
    movlw   ALERT_KEY_PRESS
    movwf   tuneCurrent

    clrf    appDspTmr
    
    movlw   0x01                  
    addwf   numEntriesL,F
    btfsc   STATUS,C
    incf    numEntriesH,F       ; Add 1 to number of stored points
    
    movf    numEntriesH, W
    movwf   eepData
    movlw   0x02
    movwf   eepAddL
    clrf    eepAddH
    call    EEPWrite
    movf    numEntriesL, W
    movwf   eepData
    call    EEPWrite            ; store number of stored points
    
    ; Calculate the address for this entry:
    ; ( numEntries * 6 ) + 2
    ; numEntriesH * 6 * 256 + ( numEntriesL * 6 )
    movlw   0x06
    mulwf   numEntriesH
    movf    PRODL, W    
    movwf   eepAddH 
    movlw   0x02
    movwf   eepAddL
    movlw   0x06
    mulwf   numEntriesL
    movf    PRODL, W
    addwf   eepAddL, F
    movf    PRODH, W
    addwfc  eepAddH, F
    
    ; Now store the data. This is
;            binary minutes, signed lat          3
;            binary minutes, signed long         3

    movf    Latmmhh, W
    movwf   eepData
    call    EEPWrite
    movf    Latmmhh+1, W
    movwf   eepData
    call    EEPWrite
    movf    Latmmhh+2, W
    movwf   eepData
    call    EEPWrite
    movf    Longmmhh, W
    movwf   eepData
    call    EEPWrite
    movf    Longmmhh+1, W
    movwf   eepData
    call    EEPWrite
    movf    Longmmhh+2, W
    movwf   eepData
    call    EEPWrite
    
    return
    
    

;*****************************************************************************        
;
;   Function :  SelectKeyPressed
;               This is the 'default' handler for when the select key is 
;               pressed. It will be call by most display handlers, but not
;               necessarily all.
;       
;               It records the current display as the default in the future.
;
;   Input:      None.
;
;   Output:     None.
;
;*****************************************************************************        
SelectKeyPressed    
    movlw   ALERT_KEY_PRESS
    movwf   tuneCurrent

    clrf    appDspTmr   ; Ensure the new display is shown immediately

    bcf     flags2, KEY_SELECT_FLAG2

    ; Stop at first entry
    movf    defaultDisplay, W
    btfsc   STATUS, Z
    return
    
    ; decrement by 1
    decf    defaultDisplay, F
    movf    defaultDisplay, W
    
    movwf   eepData
    clrf    eepAddH
    movlw   0x04
    movwf   eepAddL
    call    EEPWrite           

    return



;*****************************************************************************        
;
;   Function :  NextKeyPressed
;               Cycles to the next display in the list
;
;   Input:      None.
;
;   Output:     None.
;
;*****************************************************************************        
NextKeyPressed    
    movlw   ALERT_KEY_PRESS
    movwf   tuneCurrent

; From version 1.4, there is no concept of a 'temporary' display    
;    movlw   D'100'
;    movwf   appTmpDspTmr    ; Allow temporary displays to be active for 5s
;    bsf     flags1, TMPDSP_FLAG1
    
    clrf    appDspTmr   ; Ensure the new display is shown immediately

    bcf     flags2, KEY_NEXT_FLAG2
    
    ; Move to the next display
    incf    defaultDisplay, F
    
    movf    defaultDisplay, W
    sublw   NUM_DISPLAYS
    btfsc   STATUS, Z
    clrf    defaultDisplay   

    ; Store the new current display
    movf    defaultDisplay, W
    movwf   eepData
    clrf    eepAddH
    movlw   0x04
    movwf   eepAddL
    call    EEPWrite           

    return
    
    

;*****************************************************************************        
;
;   Function :  AppHWInit
;               Perform the hardware initialisation required by the application
;
;   Input:      None.
;
;   Output:     None
;
;*****************************************************************************        
AppHWInit
    clrf    TBLPTRU
    
    ; Init the keyboard interface variable
    clrf    keyStates       ; No key pressed

    clrf    tuneCurrent     ; Zero tune - no tune
    clrf    intTuneCurrent    

    ; Configure the GPS serial port - interrupt
    ; 4800, N, 8, 1
    
    movlw   PORTC_APP_TRIS_VAL
    movwf   TRISC 
    
    movlw   HIGH GPS_BAUDRATE
    movwf   SPBRGH
    movlw   LOW GPS_BAUDRATE
    movwf   SPBRG

    bsf     BAUDCON, BRG16
    bsf     TXSTA, BRGH
    bcf     TXSTA, SYNC
    bsf     RCSTA, SPEN
    bsf     RCSTA, CREN
    
    ; configure the SPI bus to the serial devices
    call    EEPHWInit

    clrf    INTCON  
    clrf    INTCON3
    clrf    PIE1
    clrf    PIE2
    
    ; The sound generator interrupt. 
    bsf     INTCON, TMR0IE
    
    ; 50ms tick timer on timer 3 (ccp2)
    clrf    TMR3H
    clrf    TMR3L
    
    movlw   0x39        ; 00111001
    movwf   T3CON
        
    movlw   HIGH D'62500'
    movwf   CCPR2H
    movlw   LOW D'62500'
    movwf   CCPR2L
    
    movlw   0x0B          ; 00001011
    movwf   CCP2CON
    
    ; Enable receiver interrupts
    
    bsf     PIE1, RCIE
    
    clrf    rxErrCount
    clrf    flags2
    clrf    keyNextTmr
    clrf    keySelectTmr
    clrf    keyClearTmr
    clrf    keyRecordTmr
    
    clrf    Alt1
    clrf    Alt2
    clrf    Alt3
    clrf    Alt4
    clrf    Head1, BANKED
    clrf    Head2, BANKED
    clrf    Head3, BANKED
    clrf    SpeedH
    clrf    SpeedL
    clrf    flags1
    clrf    appDspTmr   
;    clrf    appTmpDspTmr  
        
    movlw   ' '
    movwf   latText,       BANKED
    movwf   latText+1,     BANKED
    movwf   latText+2,     BANKED
    movwf   latText+3,     BANKED
    movwf   latText+4,     BANKED
    movwf   latText+5,     BANKED
    movwf   latText+6,     BANKED
    movwf   latText+7,     BANKED
    movwf   latText+8,     BANKED
    movwf   latText+9,     BANKED
    movwf   latText+0x0A,  BANKED
    movwf   longText,      BANKED
    movwf   longText+1,    BANKED
    movwf   longText+2,    BANKED
    movwf   longText+3,    BANKED
    movwf   longText+4,    BANKED
    movwf   longText+5,    BANKED
    movwf   longText+6,    BANKED
    movwf   longText+7,    BANKED
    movwf   longText+8,    BANKED
    movwf   longText+9,    BANKED
    movwf   longText+0x0A, BANKED
    clrf    latText+0x0B,  BANKED
    clrf    longText+0x0B, BANKED
    
    ; enable ccp2 software interrupts
    bsf     PIE2, CCP2IE         ; enable ccp2 int
    bsf     INTCON, PEIE
    bsf     INTCON, GIE
    
    
    
    ; Enable Timer 1 as a free running 0.8ms timer
    ; used for measuring the time it takes to scan the database
    ; movlw   0xB1  ; 10110001  I-clk, 1/8, run
    ; movwf   T1CON
    
    
        
    return
    


;*****************************************************************************        
;
;   Function :  DspHexByte
;               Write a 2 digit hex byte in uppercase to the display
;
;   Input:      byte to display in W
;
;   Output:     2 characters written to the display
;
;*****************************************************************************        
DspHexByte    
    movwf   tmpHexByte
    swapf   tmpHexByte,W
    sublw   0x09
    swapf   tmpHexByte,W
    andlw   0x0F
    btfss   STATUS,DC
    addlw   'A' - .10 - '0'
    addlw   '0'
    call    DspPutChar
    movf    tmpHexByte, W
    sublw   0x09
    movf    tmpHexByte, W
    andlw   0x0F
    btfss   STATUS,DC
    addlw   'A' - .10 - '0'
    addlw   '0'
    call    DspPutChar
    return



;*****************************************************************************        
;
;   Function :  InitDspBarSet
;               Download the special graphics characters to the LCD for 
;               forming the distance bar graph
;
;   Input:      None.
;
;   Output:     None
;
;*****************************************************************************        
InitDspBarSet
    movlw   0x40
    call    DspPutCmd       ; Point to character graphic ram

    DISPLAY STRCharData     ; Write the character generator data

    movlw   0x80
    call    DspPutCmd       ; Point to character ram
    
    
    return
    
    
    
;*****************************************************************************        
;
;   Function :  UIDspStr
;               Write a string to the display at the current cursor position.
;               This does not handle line wrapping
;
;   Input:      TBLPTRH/L setup with string address.
;
;   Output:     None
;
;*****************************************************************************        
UIDspStr
UID001
    tblrd*+
    movf    TABLAT, W
    bz      UID000
    call    DspPutChar
    bra     UID001    
    
UID000
    return



STRAcqGPSL1
    DB      "Acquiring GPS",0
STRAcqGPSL2
    DB      "Satellite data",0    
STREEPErase
    DB      "Init EEPROM...",0    
STRSWVL1
    DB      "S/W Version 1.4",0
STRSWVL2
    DB      "Date: 31/01/06",0
STREntries
    DB      "Entries",0 
STRResetMemoryL1
    DB      "Clear All",0
STRResetMemoryL2
    DB      "Entries?",0
STRMemoryResetL1
    DB      "All entries",0
STRMemoryResetL2
    DB      "removed.",0
STRConnectPCL1
    DB      "Connect to PC?",0
STRConnectPCL2
    DB      "Press 'Next'  ",0
STRConnectedPCL1
    DB      "Connected to PC",0
STRNoLock
    DB      "  No GPS Signal ",0
STRDistance
    DB      "Camera Proximity",0
STRNoCameras
    DB      "No Camera Nearby",0
STR30MPH
    DB      "   30MPH LIMIT  ",0
STR40MPH
    DB      "   40MPH LIMIT  ",0
STR50MPH
    DB      "   50MPH LIMIT  ",0
STR60MPH
    DB      "   60MPH LIMIT  ",0
STR70MPH
    DB      "   70MPH LIMIT  ",0
STR30KPH
    DB      "  30KM/H LIMIT  ",0
STR50KPH
    DB      "  50KM/H LIMIT  ",0
STR80KPH
    DB      "  80KM/H LIMIT  ",0
STR100KPH
    DB      " 100KM/H LIMIT  ",0
STR120KPH
    DB      " 120KM/H LIMIT  ",0
STRAltitude 
    DB      "Altitude ",0
STRHeading 
    DB      "Heading ",0
STRUnknown
    DB      "Unknown",0
STRNoSpeed
    DB      "  No speed data ",0        
STRAskMelodyTest
    DB      "Run Melody Test?",0
STRMelodyTest
    DB      " Melody Tester  ",0
STRCommsPCL1    
    DB      "Exchanging Data ",0
    
; LCD bitmap string. This is a sneaky way to program the
; LCD userdefined bitmaps, by pretending to write a text
; string.
STRCharData
    DB      0x01, 0x03, 0x07, 0x0f, 0x07, 0x03, 0x01, 0x80
    DB      0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x80
    DB      0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x80
    DB      0x1c, 0x1c, 0x1c, 0x1c, 0x1c, 0x1c, 0x1c, 0x80
    DB      0x1e, 0x1e, 0x1e, 0x1e, 0x1e, 0x1e, 0x1e, 0x80
    DB      0x1f, 0x1f, 0x1f, 0x1f, 0x1f, 0x1f, 0x1f, 0x80
    DB      0x1c, 0x14, 0x1c, 0x80, 0x80, 0x80, 0x80, 0x80 ; degree symbol
    DB      0x10, 0x18, 0x1c, 0x1e, 0x1c, 0x18, 0x10, 0x80
    
    DB      0x00 


