;*****************************************************************************        
;
;   Module:     flash.inc
;               
;   Author:     Mike Hibbett 
;                                                                  
;   Version:    1.0 9/12/04                                                  
;
;               Functions to implement flash programming via a bit bashed
;               serial port
;               
;               RC4 = Key used to trigger flash download
;
;               RA0 = /WE
;               RA1 = /CE
;               RA2 = TX out of PIC 
;               RA3 = /OE
;               RA4 = RX into PIC
;               RA5 = KEYEN ( keep to zero )
;
;*****************************************************************************        


TX_DATA         EQU 2
RX_DATA         EQU 4
RS232_PORT      EQU PORTA

CMD_QUESTION    EQU 0x51        ; Command byte to Host: What option?
CMD_ERASE       EQU 0x62        ; Erase chip
CMD_PROG        EQU 0x73        ; Program 512K
CMD_VERIFY      EQU 0x84        ; Validate 512K
CMD_VERIFY_RES  EQU 0x56        ; 'V' letter response
CMD_ERASE_RES   EQU 0x45        ; 'E' letter response
CMD_PROG_RES    EQU 0x50        ; 'P' letter response

; We will never return to the main application, so with the exception
; of the delay variables we can re-map the RAM variables to our own use
portaCopy       EQU 0x24        ; control lines interface mirror
hostByte        EQU 0x25        ; data from host
addCountL       EQU 0x26  
addCountM       EQU 0x27
addCountH       EQU 0x28
addrL           EQU 0x29  
addrM           EQU 0x2A
addrH           EQU 0x2B  
fdata           EQU 0x2C
rsTmp           EQU 0x2D
bitDelay        EQU 0x2E
bitCount        EQU 0x2F    
    
;*****************************************************************************        
;
;   Function :  initHW4Flash
;               Setup the hardware for flash programming; 
;               LCD is not connected, RS232 is
;          
;   Input:      None
;
;   Output:     None
;
;*****************************************************************************        
initHW4Flash
    ; Turn off keyboard, WE = OE = CE = 1 TXD = 1
    ; TRISC ouput ( for writing data to flash )

    movlw   0x0F            
    movwf   PORTA
    movwf   portaCopy
    bsf     STATUS, RP0     ; Select bank 1
    movlw   0x000           ; Set outputs
    movwf   TRISC
    bcf     STATUS, RP0     ; Select bank 0

    ; delay a bit, to allow the PC to accept the RS232 signal
    ; (The setup may have caused random data to appear at the PC)
    movlw   D'5'
    call    uiWait10ms
    
    return



;*****************************************************************************        
;
;   Function :  waitBitTime
;               Waits for 1 bit time
;               Baud rate is 38400 @ 20MHz
;
;   Input:      None
;
;   Output:     None
;  
;
;*****************************************************************************        
waitBitTime
    ; Including the call and return overhead (800ns), delay 26us
    ; 1 cycle = 200ns

    ; 25.2us delay coming up..
    ; 0.4 + (42 - 1) * 0.6 + 0.2
    
    movlw   D'42'           ; 200ns
    movwf   bitDelay        ; 200ns
wbt001    
    decfsz  bitDelay,F      ; 200 (400)ns
    goto    wbt001          ; 400ns

    return


;*****************************************************************************        
;
;   Function :  waitTxBitTime
;               Waits for 1 bit time, less a constant time of 1.8us
;               Baud rate is 38400 @ 20MHz
;
;   Input:      None
;
;   Output:     None
;  
;
;*****************************************************************************        
waitTxBitTime
    ; Including the call and return overhead (800ns), delay 26us - 1,8us
    ; 1 cycle = 200ns

    ; 25.2us delay coming up..
    ; 0.4 + (39 - 1) * 0.6 + 0.2
    
    movlw   D'39'           ; 200ns
    movwf   bitDelay        ; 200ns
wtbt001    
    decfsz  bitDelay,F      ; 200 (400)ns
    goto    wtbt001          ; 400ns

    return



;*****************************************************************************        
;
;   Function :  waitHalfBitTime
;               Waits for 1 bit time
;               Baud rate is 38400 @ 20MHz
;
;   Input:      None
;
;   Output:     None
;  
;
;*****************************************************************************        
waitHalfBitTime
    ; Including the call and return overhead (800ns), delay 13us
    ; 1 cycle = 200ns

    ; 12.2us delay coming up..
    ; 0.4 + (42 - 1) * 0.6 + 0.2
    
    movlw   D'20'           ; 200ns
    movwf   bitDelay        ; 200ns
whbt001    
    decfsz  bitDelay,F      ; 200 (400)ns
    goto    whbt001         ; 400ns
    
    nop
    
    return


;*****************************************************************************        
;
;   Function :  RS232Exchange
;               Sends byte in hostByte to host, waits (forever) for a reply
;               Baud rate is 38400 @ 20MHz
;   Input:      byte in W
;
;   Output:     response in hostByte
;  
;               NOTE: This function returns before the stop bit from the
;               host has completed.
;
;*****************************************************************************        
RS232Exchange
    movfw   hostByte
    movwf   rsTmp   
    
    movlw   0x08
    movwf   bitCount
    
    ; start bit
    bcf     portaCopy, TX_DATA
    movfw   portaCopy
    movwf   RS232_PORT

re001    
    call    waitTxBitTime             ; wait 26us - 1.8us
    
    rrf     rsTmp,F
    bcf     portaCopy, TX_DATA
    btfsc   STATUS,C
    bsf     portaCopy, TX_DATA
    movfw   portaCopy
    movwf   RS232_PORT
    decfsz  bitCount,F
    goto    re001
    
    call    waitTxBitTime             ; wait 26us - 1.8us
    nop
    nop
    nop
    nop
    
    ; stop bit
    bsf     portaCopy, TX_DATA
    movfw   portaCopy
    movwf   RS232_PORT
    
    ; Now look for a reply, store in hostByte
    
    movlw   0x08
    movwf   bitCount
    clrf    hostByte
    
re002
    btfsc   RS232_PORT, RX_DATA
    goto    re002
    
    call    waitBitTime         ; Past the start bit
    call    waitHalfBitTime     ; to middle of first bit
    
re003
    bsf     STATUS,C
    btfss   RS232_PORT, RX_DATA
    bcf     STATUS,C
    rrf     hostByte, F
    call    waitTxBitTime             ; wait 26us - 1.8us
    nop
    nop
    decfsz  bitCount,F
    goto    re003
        
    
    
    
    return


;*****************************************************************************        
;
;   Function :  updateFlash
;               downloads the 512K flash image through a bit-bash serial port
;          
;   Input:      None
;
;   Output:     None
;
;*****************************************************************************        
updateFlash  
    ; Setup the hardware for flash programming
    call    initHW4Flash    
    
    ; ask PC for an option to execute
    movlw   CMD_QUESTION
    movwf   hostByte
    
up001    
    call    RS232Exchange
    movlw   D'100'
    call    uiWait10ms
    
    movlw   CMD_VERIFY_RES
    subwf   hostByte,W
    btfsc   STATUS,Z
    goto    verify
    movlw   CMD_PROG_RES
    subwf   hostByte,W
    btfsc   STATUS,Z
    goto    prog
    movlw   CMD_ERASE_RES
    subwf   hostByte,W
    btfsc   STATUS,Z
    goto    erase
    goto    up001
     
     
;*****************************************************************************        
;
verify
    ; Data bus must be an input for PROM reading   
    bsf     STATUS, RP0     ; Select bank 1
    movlw   0x0FF           ; Set inputs
    movwf   DATA_BUS_TRIS
    bcf     STATUS, RP0     ; Select bank 0

    clrf    PROM_ADD_L
    clrf    PROM_ADD_M
    clrf    PROM_ADD_H
    clrf    promAddL
    clrf    promAddM
    clrf    promAddH


    bsf     portaCopy, ROM_WE
    movfw   portaCopy
    movwf   CONTROL
    
    bcf     portaCopy, ROM_CE
    movfw   portaCopy
    movwf   CONTROL

v001
    bcf     portaCopy, ROM_OE
    movfw   portaCopy
    movwf   CONTROL
    movfw   DATA_BUS
    
    movwf   hostByte
    
    bsf     portaCopy, ROM_OE
    movfw   portaCopy
    movwf   CONTROL
    INCPINS
    
    call    waitBitTime
    call    RS232Exchange
    
    ; PC is too slow to keep up
;    movlw   0x0A
;    call    uiWait100us

    incfsz  promAddL,F
    goto    $+4
    incfsz  promAddM,F
    goto    $+2
    incf    promAddH,F

    btfss   promAddH, 3
    goto    v001

    ; Finished, so disable flash          
    bsf     portaCopy, ROM_CE
    movfw   portaCopy
    movwf   CONTROL

    ; Data bus must be an output normally   
    bsf     STATUS, RP0     ; Select bank 1
    movlw   0x000           ; Set inputs
    movwf   DATA_BUS_TRIS
    bcf     STATUS, RP0     ; Select bank 0

    goto    up001     

;*****************************************************************************        
;
prog
    clrf    promAddL
    clrf    promAddM
    clrf    promAddH

p001    
    call    RS232Exchange

    call    writeFlashUnlock

    ; flash address 0x555    
    movlw   0x01
    movwf   PROM_ADD_L
    movlw   0x4B
    movwf   PROM_ADD_M
    movlw   0x02
    movwf   PROM_ADD_H

    movlw   0xA0
    movwf   fdata
    call    writeFlashCmd

    movfw   promAddL
    movwf   PROM_ADD_L
    movfw   promAddM
    movwf   PROM_ADD_M
    movfw   promAddH
    movwf   PROM_ADD_H
    movfw   hostByte
    movwf   fdata 
    call    writeFlashCmd
 
    incfsz  promAddL,F
    goto    $+4
    incfsz  promAddM,F
    goto    $+2
    incf    promAddH,F

    btfss   promAddH, 3
    goto    p001
    goto    up001
    

;*****************************************************************************        
;
erase
    call    writeFlashUnlock

    ; flash address 0x555    
    movlw   0x01
    movwf   PROM_ADD_L
    movlw   0x4B
    movwf   PROM_ADD_M
    movlw   0x02
    movwf   PROM_ADD_H

    movlw   0x80
    movwf   fdata
    call    writeFlashCmd
    movlw   0xAA
    movwf   fdata
    call    writeFlashCmd
        
    ; flash address 0x2AA    
    movlw   0x00
    movwf   PROM_ADD_L
    movlw   0x94
    movwf   PROM_ADD_M
    movlw   0x05
    movwf   PROM_ADD_H

    movlw   0x55
    movwf   fdata
    call    writeFlashCmd

    ; flash address 0x555    
    movlw   0x01
    movwf   PROM_ADD_L
    movlw   0x4B
    movwf   PROM_ADD_M
    movlw   0x02
    movwf   PROM_ADD_H

    movlw   0x10
    movwf   fdata
    call    writeFlashCmd

    ; Wait for 64s!

    movlw   D'65'
    movwf   fdata
e001    
    movlw   D'100'
    call    uiWait10ms
    decfsz  fdata,F
    goto    e001

    goto    up001     
     


;*****************************************************************************        
;
;   Function :  writeFlashCmd
;               Writes a byte to the flash; Used in the command sequence
;               and also for general programming
;          
;   Input:      Address on port pins, data to write in fdata
;               Data bus already set to output
;
;   Output:     None
;
;*****************************************************************************        
writeFlashCmd
    ; CE, OE, WE high
    ; Set address to 0x555 - on chip!!
    ; Set data to 0xAA
    ; CE, low
    ; WE low, then high
    ; address = 0x2AA
    ; CE high
    ; data to 0x55
    ; CE low
    ; WE low, then high
    ; CE high

    bsf     portaCopy, ROM_WE
    bsf     portaCopy, ROM_OE
    bsf     portaCopy, ROM_CE
    movfw   portaCopy
    movwf   CONTROL

    movfw   fdata
    movwf   DATA_BUS

    bcf     portaCopy, ROM_CE
    movfw   portaCopy
    movwf   CONTROL

    bcf     portaCopy, ROM_WE
    movfw   portaCopy
    movwf   CONTROL

    bsf     portaCopy, ROM_WE
    movfw   portaCopy
    movwf   CONTROL

    bsf     portaCopy, ROM_CE
    movfw   portaCopy
    movwf   CONTROL

    return
    
    
;*****************************************************************************        
;
;   Function :  writeFlashUnlock
;               Writes the unclock command sequence
;          
;   Input:      Data bus already set to output
;
;   Output:     None, fdata modified
;
;*****************************************************************************        
writeFlashUnlock
    ; CE, OE, WE high
    ; Set address to 0x555 - on chip!!
    ; Set data to 0xAA
    ; CE, low
    ; WE low, then high
    ; address = 0x2AA
    ;  CE high
    ; data to 0x55
    ; CE low
    ; WE low, then high
    ; CE high

    bsf     portaCopy, ROM_WE
    bsf     portaCopy, ROM_OE
    bsf     portaCopy, ROM_CE
    movfw   portaCopy
    movwf   CONTROL

    ; flash address 0x555    
    movlw   0x01
    movwf   PROM_ADD_L
    movlw   0x4B
    movwf   PROM_ADD_M
    movlw   0x02
    movwf   PROM_ADD_H

    movlw   0xAA
    movwf   DATA_BUS

    bcf     portaCopy, ROM_CE
    movfw   portaCopy
    movwf   CONTROL

    bcf     portaCopy, ROM_WE
    movfw   portaCopy
    movwf   CONTROL

    bsf     portaCopy, ROM_WE
    movfw   portaCopy
    movwf   CONTROL

    ; flash address 0x2AA    
    movlw   0x00
    movwf   PROM_ADD_L
    movlw   0x94
    movwf   PROM_ADD_M
    movlw   0x05
    movwf   PROM_ADD_H

    bsf     portaCopy, ROM_CE
    movfw   portaCopy
    movwf   CONTROL

    movlw   0x55
    movwf   DATA_BUS

    bcf     portaCopy, ROM_CE
    movfw   portaCopy
    movwf   CONTROL

    bcf     portaCopy, ROM_WE
    movfw   portaCopy
    movwf   CONTROL

    bsf     portaCopy, ROM_WE
    movfw   portaCopy
    movwf   CONTROL

    bsf     portaCopy, ROM_CE
    movfw   portaCopy
    movwf   CONTROL

    return
        