;*****************************************************************************        
;
;   Module:     PC.inc
;               
;   Author:     Mike Hibbett 
;                                                                  
;   Version:    1.0 5/10/03                                                  
;
;               Contains all the code for configuring and communicating
;               over the RS232 port
;
;*****************************************************************************        



;*****************************************************************************        
;
;   Function :  pcPwrOn
;               Powers up the RS232 interface
;
;   Input:      None.
;
;   Output:     None.
;
;*****************************************************************************        
pcPwrOn    
    clrf    PORTC
    bsf     STATUS,RP0    
    movlw   b'11011111'
    andwf   TRISC,F
    movlw   b'10000000'
    iorwf   TRISC,F
    movlw   BAUD_CONSTANT    ;set baud rate 
    movwf   SPBRG
    bsf     TXSTA,BRGH    ;baud rate high speed option
    bsf     TXSTA,TXEN    ;enable transmission
    bcf     STATUS,RP0    
    bsf     RCSTA,CREN    ;enable reception
    bsf     RCSTA,SPEN    ;enable serial port
    
    bsf     PORTC, RS232_PWR
    
    ; MAX232 needs 50ms following power up;
    movlw   D'5'
    call    uiWait10ms
    
    ; Could  clear the rx buffer at this point
    return
    
    

;*****************************************************************************        
;
;   Function :  pcHWInit
;               pcPwrOff
;               Initialised the hardware ports for controlling the 
;               RS232 power line, and removes power.
;               Two labels provided to this function to distinguish the use.
;
;   Input:      None.
;
;   Output:     None.
;
;*****************************************************************************        
pcHWInit
pcPwrOff
    bsf     STATUS,RP0    
    movlw   b'00011111'
    andwf   TRISC,F
    bcf     TXSTA,TXEN            ;disable transmission
    bcf     STATUS,RP0    
    bcf     RCSTA,CREN            ;disable reception
    bcf     RCSTA,SPEN            ;disable serial port
    clrf    PORTC               ; RS232 Power off and pins low
    return

    

;*****************************************************************************        
;
;   Function :  
;               pcHandler
;               This function is the 'main loop' for communication over the
;               RS232 interface. The RS232 driver should have been powered 
;               up before entering. Power remains ON at exit
;
;   Input:      None.
;
;   Output:     None.
;
;*****************************************************************************        
pcHandler
    movf    RCREG, W                ; Clear any false rx'ed data
    bcf     PIR1, RCIF
    clrf    RXBufferCount
    bcf     flags2, RS232_FLAGS2    ; Init variables

pcHLoop
    ; Check for a key press - this will cancel serial comms
    call    kbdReadRaw
    movfw   rawKey
    sublw   0x0f
    btfss   STATUS, Z
    return           

pcHtest
    call    testRS232               ; Store ay incoming rs232 data
    
    btfsc   flags2, RS232_FLAGS2    ; Look to see if we have a complete line of data
    call    processRS232            ; Convert it into cmd parm1 parm2 if so
    
    goto    pcHLoop
    


;*****************************************************************************        
;
;   Function :  testRS232
;               This function collects bytes over the RS232 interface and
;               builds a commandline buffer. 
;
;   Input:      None.
;
;   Output:     RS232_FLAGS2 set when input ready for processing
;
;*****************************************************************************        
testRS232    
    btfss    PIR1,RCIF    ;check if data received
    return

    movf    RXBufferCount, W
    xorlw   D'16'
    btfsc   STATUS, Z
    bsf     flags2, RS232_FLAGS2
    btfsc   flags2, RS232_FLAGS2
    return

    movlw   RXBuffer
    movwf   FSR
    movf    RXBufferCount, W
    addwf   FSR, F
    movf    RCREG, W
    movwf   INDF
    xorlw   0x0D
    btfsc   STATUS, Z
    bsf     flags2, RS232_FLAGS2
    btfss   STATUS, Z
    incf    RXBufferCount, F
    movf    INDF, W
    call    SerialTransmit    
    return

   

;*****************************************************************************        
;
;   Function :  processRS232
;               This function parses the received data buffer, extracting 
;               a command identifier and input parameters. It then calls
;               the appropriate handler. 
;
;   Input:      None.
;
;   Output:     
;
;*****************************************************************************        
processRS232    
    movlw   RXBuffer
    movwf   FSR
    clrf    CommandParam1_H
    clrf    CommandParam1_L
    clrf    CommandParam2_H
    clrf    CommandParam2_L
    movf    INDF, W
    movwf   Command
    
pr1                            ; Find first space character
    decf    RXBufferCount, F
    btfsc   STATUS, Z    
    goto    prexit
    
    incf    FSR, F
    movf    INDF, W
    xorlw   ' '
    btfss   STATUS, Z    
    goto    pr1

pr2                            ; Find first non space
    decf    RXBufferCount, F
    btfsc   STATUS, Z    
    goto    prexit
    
    incf    FSR, F
    movf    INDF, W
    xorlw   ' '
    btfsc   STATUS, Z    
    goto    pr2
    
    call    convertINDFtoNibble ; store first character, as a number
    movwf   CommandParam1_L

    decf    RXBufferCount, F
    btfsc   STATUS, Z    
    goto    prexit
    
    incf    FSR, F
    movf    INDF, W
    xorlw   ' '
    btfsc   STATUS, Z    
    goto    p2                  ; We have a space character
    
    swapf   CommandParam1_L, F
    call    convertINDFtoNibble ; store 2nd character, as a number
    iorwf   CommandParam1_L, F

    decf    RXBufferCount, F
    btfsc   STATUS, Z    
    goto    prexit
    
    incf    FSR, F
    movf    INDF, W
    xorlw   ' '
    btfsc   STATUS, Z    
    goto    p2                  ; We have a space character

    swapf   CommandParam1_L, W
    andlw   0x0F
    movwf   CommandParam1_H
    swapf   CommandParam1_L, W
    andlw   0xF0
    movwf   CommandParam1_L
    call    convertINDFtoNibble ; store 3rd character, as a number
    iorwf   CommandParam1_L, F

    decf    RXBufferCount, F
    btfsc   STATUS, Z    
    goto    prexit
    
    incf    FSR, F
    movf    INDF, W
    xorlw   ' '
    btfsc   STATUS, Z    
    goto    p2                  ; We have a space character

    swapf   CommandParam1_H, F
    swapf   CommandParam1_L, W
    andlw   0x0F
    iorwf   CommandParam1_H, F
    swapf   CommandParam1_L, W
    andlw   0xF0
    movwf   CommandParam1_L

    call    convertINDFtoNibble ; store 4th character, as a number
    iorwf   CommandParam1_L, F
     
p2                                ; Find first non space
    decf    RXBufferCount, F
    btfsc   STATUS, Z    
    goto    prexit
    
    incf    FSR, F
    movf    INDF, W
    xorlw   ' '
    btfsc   STATUS, Z    
    goto    p2
    
    call    convertINDFtoNibble ; store first character, as a number
    movwf   CommandParam2_L

    decf    RXBufferCount, F
    btfsc   STATUS, Z    
    goto    prexit
    
    incf    FSR, F
    movf    INDF, W
    xorlw   ' '
    btfsc   STATUS, Z    
    goto    p3                  ; We have a space character
    
    swapf   CommandParam2_L, F
    call    convertINDFtoNibble ; store 2nd character, as a number
    iorwf   CommandParam2_L, F

    decf    RXBufferCount, F
    btfsc   STATUS, Z    
    goto    prexit
    
    incf    FSR, F
    movf    INDF, W
    xorlw   ' '
    btfsc   STATUS, Z    
    goto    p3                  ; We have a space character

    swapf   CommandParam2_L, W
    andlw   0x0F
    movwf   CommandParam2_H
    swapf   CommandParam2_L, W
    andlw   0xF0
    movwf   CommandParam2_L
    call    convertINDFtoNibble ; store 3rd character, as a number
    iorwf   CommandParam2_L, F

    decf    RXBufferCount, F
    btfsc   STATUS, Z    
    goto    prexit
    
    incf    FSR, F
    movf    INDF, W
    xorlw   ' '
    btfsc   STATUS, Z    
    goto    p3                  ; We have a space character

    swapf   CommandParam2_H, F
    swapf   CommandParam2_L, W
    andlw   0x0F
    iorwf   CommandParam2_H, F
    swapf   CommandParam2_L, W
    andlw   0xF0
    movwf   CommandParam2_L

    call    convertINDFtoNibble ; store 4th character, as a number
    iorwf   CommandParam2_L, F
     
p3         ; there is no third parameter, so end here.    
prexit
    movlw   0x0A
    call    SerialTransmit
    movlw   0x0D
    call    SerialTransmit

    ; call the command handler
    movfw   Command
    xorlw   'W'
    btfsc   STATUS, Z
    call    doWrite
    movfw   Command
    xorlw   'w'
    btfsc   STATUS, Z
    call    doWrite
    movfw   Command
    xorlw   'R'
    btfsc   STATUS, Z
    call    doRead
    movfw   Command
    xorlw   'r'
    btfsc   STATUS, Z
    call    doRead
    
    movlw   0x0A
    call    SerialTransmit
    movlw   0x0D
    call    SerialTransmit

    ; We have finished with the buffer, so make it available again
    bcf     flags2, RS232_FLAGS2
    clrf    RXBufferCount
    return
   
    
    
;*****************************************************************************        
;
;   Function :  SerialReceive
;               This function extracts a byte from the receiver hardware.
;               It will block until a byte is available 
;
;   Input:      None.
;
;   Output:     byte in W
;
;*****************************************************************************        
SerialReceive
    btfss   PIR1,RCIF    ;check if data received
    goto    $-1            ;wait until new data
    movf    RCREG,W        ;get received data into W
    return



;*****************************************************************************        
;
;   Function :  SerialTransmit
;               This function sends the byte in W over the RS232 port. The 
;               function will wait until previous data has been sent
;
;   Input:      Byte in W
;
;   Output:     
;
;*****************************************************************************        
SerialTransmit
    btfss   PIR1,TXIF    ;check that buffer is empty
    goto    $-1
    movwf   TXREG        ;transmit byte
    return



;*****************************************************************************        
;
;   Function :  doWrite
;               Writes a byte to the eeprom
;               A 10ms delay should occur before calling this routine again
;
;   Input:      All data supplied by CommandParms
;
;   Output:     Data Written
;
;*****************************************************************************        
doWrite
    ; format is W xxxx yy  where xxxx is address, yy is value
    ; top 2 bits are eeprom num
    ; lower 14 bits are address
    swapf   CommandParam1_H, W
    andlw   0x0C
    movwf   eepNum    
    bcf     STATUS, C
    rrf     eepNum, F
    
    movf    CommandParam1_H, W
    andlw   0x3F
    movwf   eepAddH
    movf    CommandParam1_L, W
    movwf   eepAddL
    movf    CommandParam2_L, W
    movwf   eepData
    call    eepWrite
    return
    
    
    
;*****************************************************************************        
;
;   Function :  doRead
;               Dumps 256 bytes of data from the eeprom to the RS232 port
;               in a pretty format ( 16 hex bytes on a line )
;
;   Input:      All data supplied by CommandParms
;
;   Output:     Data Read & displayed
;
;*****************************************************************************        
doRead
    ; format is R xxxx   where xxxx is address block
    ; top 2 bits are eeprom num
    ; lower 14 bits are address
    
    clrf    eepNum
    movf    CommandParam1_L, W
    movwf   eepAddH
    clrf    eepAddL
    
    clrf    pcTmp

dor001
    movfw   pcTmp
    andlw   0x0F
    btfss   STATUS, Z
    goto    dor002
    movlw   0x0A
    call    SerialTransmit
    movlw   0x0D
    call    SerialTransmit
        
dor002
    call    eepRead
    call    output_hexbyte
    incf    eepAddL, F
    decf    pcTmp, F
    btfss   STATUS, Z
    goto    dor001
    return



;*****************************************************************************        
;
;   Function :  pcWelcomeBanner
;               Sends an ASCII welcome screen to the RS232 port
;
;   Input:      None
;
;   Output:     Data displayed
;
;*****************************************************************************        
pcWelcomeBanner
    clrf    DataPointer    
        
owm_001
    movlw   HIGH welcome_table
    movwf   PCLATH    
    movf    DataPointer, W
    call    welcome_table
    xorlw   0
    btfsc   STATUS, Z
    return
    
    call    SerialTransmit
    incf    DataPointer, F
    goto    owm_001

welcome_table
    addwf   PCL, F
    retlw    'C'
    retlw    'o'
    retlw    'm'
    retlw    'm'
    retlw    'a'
    retlw    'n'
    retlw    'd'
    retlw    's'
    retlw    ' '
    retlw    'a'
    retlw    'r'
    retlw    'e'
    retlw    .13
    retlw   .10
    retlw    'R'
    retlw    ' '
    retlw    'x'
    retlw    ' '
    retlw    ' '
    retlw    'R'
    retlw    'e'
    retlw    'a'
    retlw    'd'
    retlw    ' '
    retlw    'a'
    retlw    ' '
    retlw    'b'
    retlw    'l'
    retlw    'o'
    retlw    'c'
    retlw    'k'
    retlw    .13
    retlw   .10
    retlw    'W'
    retlw    ' '
    retlw    'x'
    retlw    ' '
    retlw    'b'
    retlw    ' '
    retlw    ' '
    retlw    'W'
    retlw    'r'
    retlw    'i'
    retlw    't'
    retlw    'e'
    retlw    ' '
    retlw    'a'
    retlw    ' '
    retlw    'b'
    retlw    'y'
    retlw    't'
    retlw    'e'
    retlw    .13
    retlw   .10
    retlw    'B'
    retlw    'l'
    retlw    'o'
    retlw    'c'
    retlw    'k'
    retlw    's'
    retlw    ' '
    retlw    'a'
    retlw    'r'
    retlw    'e'
    retlw    ' '
    retlw    '2'
    retlw    '5'
    retlw    '6'
    retlw    .13
    retlw   .10
    retlw    'A'
    retlw    'l'
    retlw    'l'
    retlw    ' '
    retlw    'v'
    retlw    'a'
    retlw    'l'
    retlw    'u'
    retlw    'e'
    retlw    's'
    retlw    ' '
    retlw    'i'
    retlw    'n'
    retlw    ' '
    retlw    'h'
    retlw    'e'
    retlw    'x'
    retlw    .13
    retlw   .10
welcome_table_end
    retlw    0
    
    IF ( (welcome_table & 0x0FF) >= (welcome_table_end & 0x0FF) )
        MESSG   "Table welcome_table overflow"
    ENDIF



;*****************************************************************************        
;
;   Function :  output_hexbyte
;               Outputs the byte in eepData over the RS232 port in ASCI hex
;
;   Input:      data in eepData
;
;   Output:     Data displayed
;
;*****************************************************************************        
output_hexbyte
    swapf   eepData,W
    sublw   0x09
    swapf   eepData,W
    andlw   0x0F
    btfss   STATUS,DC
    addlw   'A' - .10 - '0'
    addlw   '0'
    call    SerialTransmit
    movfw   eepData
    sublw   0x09
    movfw   eepData
    andlw   0x0F
    btfss   STATUS,DC
    addlw   'A' - .10 - '0'
    addlw   '0'
    call    SerialTransmit
    return

  
;*****************************************************************************        
;
;   Function :  convertINDFtoNibble
;               This function converts an ASCII hex character into a binary
;               nibble value.
;
;   Input:      Byte pointed to by INDF
;
;   Output:     Nibble in LSN of W
;
;*****************************************************************************        
convertINDFtoNibble
    movf    INDF, W
    sublw   0x39
    movf    INDF, W
    andlw   0x0F
    btfss   STATUS, C
    addlw   0x09
    return
  


    