Attribute VB_Name = "modPnMAlgorithm"
' TK3 programming function for new PIC types introduced by PIC n' Mix
' decoupled from the rest of TK3 to ease maintenance
' Written by Andrew Jarvis 15.08.04
'
' Revision History
' 26.03.04 Initial Revision
' 15.08.04 Final Revision
'
' Adds support for the following devices:
'
' PIC16F873A PIC16F874A
' PIC16F876A PIC16F877A
' PIC12F629  PIC12F675
' PIC16F630  PIC16F676

Option Explicit

'masks for device ID values
Private Const maskPIC12F629 = &HF80
Private Const maskPIC12F675 = &HFC0
Private Const maskPIC16F630 = &H10C0
Private Const maskPIC16F676 = &H10E0
Private Const maskPIC16F873A = &HE40
Private Const maskPIC16F874A = &HE60
Private Const maskPIC16F876A = &HE00
Private Const maskPIC16F877A = &HE20
Public Function PnM_PIC() As Boolean
'determines if PIC should be handled by this extension module
    PnM_PIC = _
    PnM_LowPinCount Or _
    PnM_16f87xA
End Function
Public Function PnM_ClassicLowPinCount() As Boolean
'determines if PIC is a classic low pin count device
    PnM_ClassicLowPinCount = _
    PICdevice = "PIC12F629" Or _
    PICdevice = "PIC12F675" Or _
    PICdevice = "PIC16F630" Or _
    PICdevice = "PIC16F676"
End Function
Public Function PnM_EnhancedLowPinCount() As Boolean
'determines if PIC is an enhanced low pin count device
    PnM_EnhancedLowPinCount = _
    PICdevice = "PIC12F635" Or _
    PICdevice = "PIC12F683" Or _
    PICdevice = "PIC16F636" Or _
    PICdevice = "PIC16F684" Or _
    PICdevice = "PIC16F688"
End Function
Public Function PnM_LowPinCount() As Boolean
'only support classic for this release
    PnM_LowPinCount = PnM_ClassicLowPinCount
End Function
Public Function PnM_16f87xA() As Boolean
'identify 16f87xA family
    PnM_16f87xA = _
    PICdevice = "PIC16F873A" Or _
    PICdevice = "PIC16F874A" Or _
    PICdevice = "PIC16F876A" Or _
    PICdevice = "PIC16F877A"
End Function
Public Function PnM_ProgramPIC(Optional ByVal silent As Boolean = False) As Boolean

    'first check to see if the PIC is recognised
    If VerifyPIC Then
        'preload the hex to memory
        PnM_LoadHexFile
        'determine correct algorithm
        If PnM_16f87xA Then
            Algorithm16f87xA
        ElseIf PnM_LowPinCount Then
            AlgorithmLowPinCount silent
        End If
        PnM_ProgramPIC = True
    Else
        PnM_ProgramPIC = False
    End If
    
End Function
Private Sub Algorithm16f87xA()
  
    Dim intLastAddress
    Dim X As Integer
    
    'put the PIC into programming mode
    PnM_EnterProgramMode
    
    'determine the last address of the PIC that will be written
    intLastAddress = PnM_LastAddress(8)
    
    ' update the gui
    TK3ProgramPIC.ProgressBar1.Max = intLastAddress + eeprom%
    TK3ProgramPIC.ProgressBar1.value = 1
    DoEvents
     
    'write program memory.
    For X = 0 To intLastAddress Step 2
        PnM_WriteProgramMemory ProgMem(X).data, ProgMem(X + 1).data, (X + 2) Mod 16 = 0
        IncProgress 2
    Next X

    ' Write configuration memory
    If ConfigMem(WORD_MSB).dirty Or ConfigMem(WORD_LSB).dirty Then
        ' write the new config memory to the device
        PnM_WriteConfiguration ConfigMem(WORD_LSB).data, ConfigMem(WORD_MSB).data
    End If

    ' set configuration mode
    PnM_EnterConfigurationMode
    'write ID locations if they exist
    For X = 0 To 3
        ' write the data at the current location
        If IDMem(WORD_MSB).dirty Or IDMem(WORD_LSB).dirty Then
            PnM_WriteProgramMemory IDMem(WORD_LSB).data, IDMem(WORD_MSB).data
        End If
        IncProgress
    Next X
                                   
    'exit configuration mode
    PnM_ExitProgramMode
    PnM_EnterProgramMode
                                   
    ' Write EE memory.
    ' In line with existing TK3 behaviour, EEPROM memory is not erased when the PIC
    ' is re-programmed. Instead, new data is overwrites old at relevant position
    For X = 0 To (eeprom% - 1)
        ' write the data at the current location
        If EEPROMMem(X).dirty Then PnM_WriteDataMemory EEPROMMem(X).data
        IncProgress
    Next X
                      
    PnM_ExitProgramMode
       
End Sub
Private Sub AlgorithmLowPinCount(ByVal silent As Boolean)
  
    'last used address of program memory
    Dim intLastAddress
    Dim X As Integer
    Dim oscL As Byte
    Dim oscH As Byte
    Dim configL As Byte
    Dim configH As Byte
    Dim bandGap As Byte
    
    'put the PIC into programming mode
    PnM_EnterProgramMode
    
    'determine the last address of the PIC that will be written
    '*excluding* calibration memory
    intLastAddress = PnM_LastAddress(1, True)
    
    ' update the gui
    If silent = False Then
        TK3ProgramPIC.ProgressBar1.Max = intLastAddress + eeprom%
        TK3ProgramPIC.ProgressBar1.value = 1
        DoEvents
    End If
    
    'read OSCCAL
    PnM_ReadOscCal (PnM_HexSize \ 2) - 1, oscL, oscH
    'exit configuration mode
    
    'if the config word will be updated, save the BG bits...
    If ConfigMem(WORD_MSB).dirty Or ConfigMem(WORD_LSB).dirty Then
        'read the configuration word
        PnM_ReadConfiguration configL, configH
        'strip out bits 13 and 14 (BandGap) - this is actually bits
        '4 and 5 of the high byte
        bandGap = configH And &H30
        'merge this with the new config word
        ConfigMem(WORD_MSB).data = ConfigMem(WORD_MSB).data Or bandGap
        'exit configuration mode
        PnM_ExitProgramMode
        PnM_EnterProgramMode
    End If

    'bulk erase required
    BulkEraseProgram
    
    'write program memory.
    For X = 0 To intLastAddress Step 2
        PnM_WriteProgramMemory ProgMem(X).data, ProgMem(X + 1).data
        If silent = False Then IncProgress 2
    Next X
    
    'reset PC
    PnM_ExitProgramMode
    PnM_EnterProgramMode
    'write OSCCAL to hex at last (calib) location
    PnM_RestoreOscCal (PnM_HexSize \ 2) - 1, oscL, oscH
    
    ' Write configuration memory
    If ConfigMem(WORD_MSB).dirty Or ConfigMem(WORD_LSB).dirty Then
        ' write the new config memory to the device
        PnM_WriteConfiguration ConfigMem(WORD_LSB).data, ConfigMem(WORD_MSB).data
    End If

    ' set configuration mode
    PnM_EnterConfigurationMode
    'write ID locations if they exist
    For X = 0 To 3
        ' write the data at the current location
        If IDMem(WORD_MSB).dirty Or IDMem(WORD_LSB).dirty Then
            PnM_WriteProgramMemory IDMem(WORD_LSB).data, IDMem(WORD_MSB).data
        End If
        If silent = False Then IncProgress
    Next X
                                   
    'exit configuration mode
    PnM_ExitProgramMode
    PnM_EnterProgramMode
                                   
    ' Write EE memory.
    ' In line with existing TK3 behaviour, EEPROM memory is not erased when the PIC
    ' is re-programmed. Instead, new data is overwrites old at relevant position
    For X = 0 To (eeprom% - 1)
        ' write the data at the current location
        If EEPROMMem(X).dirty Then PnM_WriteDataMemory EEPROMMem(X).data
        If silent = False Then IncProgress
    Next X
                      
    PnM_ExitProgramMode
       
End Sub
Private Sub IncProgress(Optional ByVal increment = 1)
    
    'updates the progress bar during PIC programming by specified increment
    Dim intNew As Integer
    intNew = TK3ProgramPIC.ProgressBar1.value + increment
    
    If intNew > TK3ProgramPIC.ProgressBar1.Max Then
        intNew = TK3ProgramPIC.ProgressBar1.Max
    End If
        
    TK3ProgramPIC.ProgressBar1.value = intNew
    
End Sub
Private Function VerifyPIC() As Boolean

    Dim actualPIC As String
        
    actualPIC = PnMQueryPIC
    If actualPIC <> PICdevice Then
        MsgBox "Expected " & PICdevice & " but found " & actualPIC & ". Please replace.", _
        vbExclamation, "PnM Extension Algorithm"
        VerifyPIC = False
    Else
        VerifyPIC = True
    End If
    
End Function

Public Function PnMQueryPIC() As String

    Dim PicType As Long
    Dim actualPIC As String
        
    'find out which PIC is on the board
    PicType = PnM_ReadPICType
    'lookup to see if it's supported by this extension module
    Select Case PicType And &H3FE0
    
    Case maskPIC12F629
        actualPIC = "PIC12F629"
    Case maskPIC12F675
        actualPIC = "PIC12F675"
    Case maskPIC16F630
        actualPIC = "PIC16F630"
    Case maskPIC16F676
        actualPIC = "PIC16F676"
    Case maskPIC16F873A
        actualPIC = "PIC16F873A"
    Case maskPIC16F874A
        actualPIC = "PIC16F874A"
    Case maskPIC16F876A
        actualPIC = "PIC16F876A"
    Case maskPIC16F877A
        actualPIC = "PIC16F877A"
    Case Else
        actualPIC = "Unknown [device ID = " & Hex(PicType) & "]"
    End Select
    
    PnMQueryPIC = actualPIC
    
End Function
