;*****************************************************************************        
;
;   Module:     fat16.inc
;               
;   Author:     Mike Hibbett, mike.hibbett@gmail.com 
;                                                                  
;   Version:    1.0 26/07/06                                              
;
;               Implementation of a FAT16 Disk File System for use with
;               Media cards ( although could be used with a real disk ) 
;
;               This code relies on just two lower level functions;
;               read512bytes and write512bytes.
;
;               It is assumed that the media has already been configured
;               and is ready to work
;
;        NOTE:  This driver currently does not support sub-directories. All
;               files must be created in the root directory.
;
;               It also does not try to fill 'holes' in the FAT table, but 
;               instead writes to the end of the 'used' memory. This 
;               increases speed, but is less efficient on storage use ( if
;               you delete files frequently )
;
;               The access functions that are defined here are simplified
;               ones. You may only have a single file open at a time.
;
;               This is example code, which does not include full error
;               checking ( for example, that there is sufficient space
;               on the media for a new file )
;
;               Although the code is fast at writing to a file ( an important
;               requirement in most cases ) it is not fast at opening or
;               closing files. This has been done for a number of reasons. For
;               example it makes the code simpler, and requires less ram space.
;
;*****************************************************************************        


;*****************************************************************************        
;   DRIVER INTERFACE
;
;   This module contains 7 interface functions that you can use to access
;   a file system on a media card. The module requires some ram, and these
;   have been defined in the file app.inc
;   
;   Before using any of the functions, you must call FAT16Init to setup 
;   various variables and read the beginning of the file system memory.
;
;   To get a directory list, first call FAT16StartDirList, then repeatedly call
;   FAT16NextDirList. The later routine will return with the Carry flag set
;   when there are no further files in the directory.
;   Each call to FAT16NextDirList will return a copy of the 32 byte directory
;   entry in FAT_Dir_Entry.
;
;   To read a file, first place the filename in the variable FAT_Filename using
;   the FAT file format ( 8 characters followed by 3 characters of the extension)
;   Then call FAT16OpenFileRead. This will return with the Carry flag set if the
;   file was not found. Otherwise, it will indicate the size of the file in
;   FAT_File_Size. To get the data, make repeated calls to FAT16ReadFile.
;   Each call will extract 512 bytes of data from the file into the sector buffer
;   at BLOCK_BUFFER_ADDRESS; you should keep a note of how many bytes you read 
;   to determine when the end of the file has been reached.
;
;   To write a file, first place the filename in the variable FAT_Filename using
;   the FAT file format ( 8 characters followed by 3 characters of the extension)
;   Then fill the 512 buffer at BLOCK_BUFFER_ADDRESS with your data.
;   Call FAT16OpenFileWrite to commit this data to the media.
;   Note: FAT16OpenFileWrite always creates a new file.
;
;   When you finish writing to the file, call FAT16CloseFileWrite. When this
;   routine returns you are free to power down the card immediately, if you
;   wish.
;
;
;*****************************************************************************        



;*****************************************************************************        
;
;   Function :  FAT16Init
;               This must be called before any of the routines.
;               It scans the MBR and FAT16 Boot record to work 
;               out where the directory table, FAT tables are, what
;               size the clusters are.
;
;
;   Input:      None.
;
;   Output:     Various variables updated
;
;               All multibyte values are stored little-endian ( LSB first )
;
;*****************************************************************************        
FAT16Init
    ; Init our variables
    clrf    FAT_flags1
    
    ; First, do we have anything that looks like a filesystem in sector 0?
    ; We should at least have an end marker. IF not, we assume a critical
    ; error in the lower layers, and exit. We ensure that the driver 
    ; indicates an error by setting a status bit.
    clrf    MMCOp1 
    clrf    MMCOp2 
    clrf    MMCOp3 
    clrf    MMCOp4 
    call    MMCReadBlock

    movff   BLOCK_BUFFER_ADDRESS + 0x1FE, WREG
    sublw   0x55
    btfss   STATUS, Z
    goto    fiNoFS

    movff   BLOCK_BUFFER_ADDRESS + 0x1FF, WREG
    sublw   0xAA
    btfss   STATUS, Z
    goto    fiNoFS

    ; OK, we have something here.
    ; Now, we need to check if the first sector on the media is the MBR
    ; or a FAT16 boot record. Based on that, get the address of the
    ; partition - in bytes. We assume 512 bytes/sector, because we have
    ; no other way of knowning at this stage. Of course if the MBR is
    ; missing then the partition starts at address 0!

    ; Is the MBR present? Assume not
    bsf     FAT_flags1, flags1_FAT_NO_MBR
    clrf    FAT_Part_Add
    clrf    FAT_Part_Add + 1
    clrf    FAT_Part_Add + 2
    clrf    FAT_Part_Add + 3

    movff   BLOCK_BUFFER_ADDRESS + 0x26, WREG
    sublw   0x29
    btfsc   STATUS, Z
    goto    fi001

    ; Yes. MBR is here. So get the partition address from it
    bcf     FAT_flags1, flags1_FAT_NO_MBR

    ; It would be a good idea to scan each partition, and
    ; not just assume ( as is normally the case ) that
    ; the first partition is valid. I do this to simplify the
    ; example.

    ; Get the sector offset to the partition.
    ; store it as a byte value, not sectors ( * 512 )
    movff   BLOCK_BUFFER_ADDRESS + 0x1BE + 8 , WREG
    movwf   FAT_Part_Add + 1
    movff   BLOCK_BUFFER_ADDRESS + 0x1BE + 9 , WREG
    movwf   FAT_Part_Add + 2
    movff   BLOCK_BUFFER_ADDRESS + 0x1BE + D'10' , WREG
    movwf   FAT_Part_Add + 3

    bcf     STATUS, C
    rlcf    FAT_Part_Add + 1, F
    rlcf    FAT_Part_Add + 2, F
    rlcf    FAT_Part_Add + 3, F
    

fi001    
    ; Finally, get the variables from the partition. Calculate the address
    ; of the first FAT table, the directory table and the start of data.
    
    ; Read the FAT16 boot record from the beginning of the partition
    movf    FAT_Part_Add+3, W
    movwf   MMCOp1 
    movf    FAT_Part_Add+2, W
    movwf   MMCOp2
    movf    FAT_Part_Add+1, W
    movwf   MMCOp3 
    movf    FAT_Part_Add, W
    movwf   MMCOp4 
    call    MMCReadBlock
    
    ; Read some important variables
    movff   BLOCK_BUFFER_ADDRESS + 0x0d , WREG
    movwf   FAT_ClusterS
    
    ; Calculate some important variables
    
    ; FAT size, in bytes
    clrf    FAT_FAT_Size
    movff   BLOCK_BUFFER_ADDRESS + 0x16 , WREG
    movwf   FAT_FAT_Size + 1
    movff   BLOCK_BUFFER_ADDRESS + 0x16 + 1, WREG
    movwf   FAT_FAT_Size + 2
    clrf    FAT_FAT_Size + 3
    bcf     STATUS, C
    rlcf    FAT_FAT_Size + 1, F
    rlcf    FAT_FAT_Size + 2, F
    rlcf    FAT_FAT_Size + 3, F
    
    ; Start address of first FAT: part start + reserved sectors
    clrf    FAT_FAT_Add
    movff   BLOCK_BUFFER_ADDRESS + 0x0E , WREG
    movwf   FAT_FAT_Add + 1
    movff   BLOCK_BUFFER_ADDRESS + 0x0E + 1 , WREG
    movwf   FAT_FAT_Add + 2
    clrf    FAT_FAT_Add + 3
    bcf     STATUS, C
    rlcf    FAT_FAT_Add + 1, F
    rlcf    FAT_FAT_Add + 2, F
    rlcf    FAT_FAT_Add + 3, F
    
    movf    FAT_Part_Add, W
    addwf   FAT_FAT_Add, F
    movf    FAT_Part_Add+1, W
    addwfc  FAT_FAT_Add+1, F
    movf    FAT_Part_Add+2, W
    addwfc  FAT_FAT_Add+2, F
    movf    FAT_Part_Add+3, W
    addwfc  FAT_FAT_Add+3, F
    
    ; Start address of directory: fatstart +  (fatsize * numfats)
    ; we are going to cheat here, and assume numfats is 2 - Microsoft claim 
    ; it is the only valid #. Odd, but true.
    movf    FAT_FAT_Size, W
    movwf   FAT_Dir_Add
    movf    FAT_FAT_Size+1, W
    movwf   FAT_Dir_Add+1
    movf    FAT_FAT_Size+2, W
    movwf   FAT_Dir_Add+2
    movf    FAT_FAT_Size+3, W
    movwf   FAT_Dir_Add+3
    bcf     STATUS, C
    rlcf    FAT_Dir_Add, F
    rlcf    FAT_Dir_Add + 1, F
    rlcf    FAT_Dir_Add + 2, F
    rlcf    FAT_Dir_Add + 3, F

    movf    FAT_FAT_Add, W
    addwf   FAT_Dir_Add, F
    movf    FAT_FAT_Add+1, W
    addwfc  FAT_Dir_Add+1, F
    movf    FAT_FAT_Add+2, W
    addwfc  FAT_Dir_Add+2, F
    movf    FAT_FAT_Add+3, W
    addwfc  FAT_Dir_Add+3, F
    
    ; Start address of data. directory start + sizeof directory
    ; directory is always 512 entries max, so we just add
    ; 512 * 32 to the start address of the directory
    ; 
    movf    FAT_Dir_Add, W
    movwf   FAT_Data_Add
    movf    FAT_Dir_Add+1, W
    movwf   FAT_Data_Add+1
    movf    FAT_Dir_Add+2, W
    movwf   FAT_Data_Add+2
    movf    FAT_Dir_Add+3, W
    movwf   FAT_Data_Add+3
    movlw   0x40
    addwf   FAT_Data_Add+1, F
    movlw   0x00
    addwfc  FAT_Data_Add+2, F
    addwfc  FAT_Data_Add+3, F
    
    goto    fiExit    

fiNoFS
    ; If an error occurred, set the flags1_FAT_INIT_ERROR bit
    bsf     FAT_flags1, flags1_FAT_INIT_ERROR

fiExit    
    ; When complete, set the flags1_FAT_INIT_COMPLETE bit
    bsf     FAT_flags1, flags1_FAT_INIT_COMPLETE
    return
    


;*****************************************************************************        
;
;   Function :  FAT16OpenFileRead
;               Locates the file on the disk, and initialises the 
;               variables associated with reading/writing to it.
;
;
;   Input:      Filename in FAT_Filename
;
;   Output:     Carry set if file not found
;
;               Note: uses FSR0
;
;               
;
;*****************************************************************************        
FAT16OpenFileRead
    ; Find file in Directory table; Dont forget this spans 32 sectors.
    ; sectors 
    ; load start cluster.
    ; record location of first free cluster to write in.
    ; Set variables to indicate a file is open.
        
    call    FAT16StartDirList
    
fof001    
    call    FAT16NextDirList
    btfsc   STATUS, C    
    return 
    
    ; Compare the current file with the one we are looking for 
    
    lfsr    FSR0, FAT_Filename
    lfsr    FSR1, FAT_Dir_Entry
    movlw   D'11'
    movwf   tmpCount

fof002    
    movf    POSTINC0, W
    subwf   POSTINC1, W
    btfss   STATUS, Z
    goto    fof001
    
    decfsz  tmpCount, F
    goto    fof002
    
    ; They are the same!
    
    ; Record the important values; FAT_File_Size, FAT_File_Ind (0), FAT_File_Pos
    
    clrf    FAT_File_Ind
    clrf    FAT_File_Ind+1
    clrf    FAT_File_Ind+2
    clrf    FAT_File_Ind+3
    
    movlw   0x08 + 1
    movwf   FAT_SecInClust
    
    movf    FAT_Dir_Entry + 0x1c, W
    movwf   FAT_File_Size
    movf    FAT_Dir_Entry + 0x1c + 1, W
    movwf   FAT_File_Size + 1
    movf    FAT_Dir_Entry + 0x1c + 2, W
    movwf   FAT_File_Size + 2
    movf    FAT_Dir_Entry + 0x1c + 3, W
    movwf   FAT_File_Size +3
        
    ; To work out the file start address, in bytes:
    ; Get the start cluster. Subtract 2 from this
    ; Multiply that by 512 * sectors per cluster
    ; Add that to FAT_Data_Add
    movf    FAT_Dir_Entry + 0x1a, W
    movwf   FAT_FileC
    movf    FAT_Dir_Entry + 0x1a + 1, W
    movwf   FAT_FileC + 1
        
    clrf    FAT_Tmp_Long, 1
    clrf    FAT_Tmp_Long+1, 1
    clrf    FAT_Tmp_Long+2, 1
    clrf    FAT_Tmp_Long+3, 1
    
    movlw   0x02
    subwf   FAT_FileC, F
    movlw   0x00
    subwfb  FAT_FileC+1, F
    
    ; If it is zero, 
    
    ; now, make FAT_Tmp_Long equal 512 * the number of sectors per cluster
    movf    FAT_ClusterS, W
    movwf   tmpCount
fof003    
    movlw   0x02
    addwf   FAT_Tmp_Long+1, F, 1
    movlw   0x00
    addwfc  FAT_Tmp_Long+2, F, 1
    addwfc  FAT_Tmp_Long+3, F, 1
    decfsz  tmpCount, F
    goto    fof003
   

    clrf    FAT_File_Pos
    clrf    FAT_File_Pos+1
    clrf    FAT_File_Pos+2
    clrf    FAT_File_Pos+3
    
    ; now multiply that by FAT_FileC, if it isn't zero
    
    movf    FAT_FileC, W
    movwf   FAT_tmp_count
    iorwf   FAT_FileC+1, W
    btfsc   STATUS, Z
    goto    fof005    
    movf    FAT_FileC+1, W
    movwf   FAT_tmp_count+1
    incf    FAT_tmp_count+1, F
    
    
fof004
    movf    FAT_Tmp_Long, W, 1
    addwf   FAT_File_Pos, F
    movf    FAT_Tmp_Long+1, W, 1
    addwfc  FAT_File_Pos+1, F
    movf    FAT_Tmp_Long+2, W, 1
    addwfc  FAT_File_Pos+2, F
    movf    FAT_Tmp_Long+3, W, 1
    addwfc  FAT_File_Pos+3, F
    
    decfsz  FAT_tmp_count, F
    goto    fof004
    decfsz  FAT_tmp_count+1, F
    goto    fof004    
     
fof005
               
    ; Then add in the data start offset
       
    movf    FAT_Data_Add, W
    addwf   FAT_File_Pos, F
    movf    FAT_Data_Add+1, W
    addwfc  FAT_File_Pos+1, F
    movf    FAT_Data_Add+2, W
    addwfc  FAT_File_Pos+2, F
    movf    FAT_Data_Add+3, W
    addwfc  FAT_File_Pos+3, F
            
    ; Signal success
    bcf     STATUS, C
    
    return



;*****************************************************************************        
;
;   Function :  FAT16OpenFileWrite
;               Creates a new file, and opens it for writing.
;
;   Input:      Filename in FAT_Filename
;
;   Output:     
;               
;
;*****************************************************************************        
FAT16OpenFileWrite

    ; TODO - Checking that the filename is valid, and that the file does not
    ; already exist, would be good at this point    

    ; We will use FAT_File_Pos as a sector pointer as we scan the FAT table
    ; for a free entry
    movf    FAT_FAT_Add, W
    movwf   FAT_File_Pos
    movf    FAT_FAT_Add+1, W
    movwf   FAT_File_Pos+1
    movf    FAT_FAT_Add+2, W
    movwf   FAT_File_Pos+2
    movf    FAT_FAT_Add+3, W
    movwf   FAT_File_Pos+3

    ; FAT_FileC will hold the cluster number we are looking for. It starts
    ; at zero, the index of the first FAT entry. We expect it to be 2 or more.
    
    clrf    FAT_FileC
    clrf    FAT_FileC + 1
    
fofw000    
    ; read the sector
    movf    FAT_File_Pos+3, W
    movwf   MMCOp1 
    movf    FAT_File_Pos+2, W
    movwf   MMCOp2
    movf    FAT_File_Pos+1, W
    movwf   MMCOp3 
    movf    FAT_File_Pos, W
    movwf   MMCOp4 
    call    MMCReadBlock
   
    lfsr    FSR0, BLOCK_BUFFER_ADDRESS

fofw001
    movf    POSTINC0, W
    iorwf   POSTINC0, W
    btfsc   STATUS, Z
    goto    fofw002             ; Found an empty entry
    
    ; Move to the next entry
    infsnz  FAT_FileC, F
    incf    FAT_FileC+1, F
    
    ; have we moved into a new sector? Read it if so
    movf    FAT_FileC, W
    btfss   STATUS, Z
    goto    fofw001
    
    movlw   0x02
    addwf   FAT_File_Pos+1, F
    movlw   0x00
    addwfc  FAT_File_Pos+2, F
    addwfc  FAT_File_Pos+3, F
    
    goto    fofw000
    
fofw002    

    ; OK, we have our cluster ID in FAT_FileC
    ; While we have that sector in memory, mark it as in use, by writing FFFF to it 
    ; This signifies that our file is in there.
    
    ; TODo - should really update the second fat table too.
    
    lfsr    FSR0, BLOCK_BUFFER_ADDRESS
    movf    FAT_FileC, W
    addwf   FSR0L, F
    movlw   0
    addwfc  FSR0H, F
    movf    FAT_FileC, W
    addwf   FSR0L, F
    movlw   0
    addwfc  FSR0H, F
    movlw   0xFF
    movwf   POSTINC0
    movwf   POSTINC0
    
    movf    FAT_File_Pos+3, W
    movwf   MMCOp1 
    movf    FAT_File_Pos+2, W
    movwf   MMCOp2
    movf    FAT_File_Pos+1, W
    movwf   MMCOp3 
    movf    FAT_File_Pos, W
    movwf   MMCOp4 
    call    MMCWriteBlock
    
    ; Find free space in the directory table ; Dont forget this spans 32 sectors.
    ; Record that cluster in the directory table entry, with the filename
    ; Zero other bits of the file entry - especially file size.
    
    movf    FAT_Dir_Add, W
    movwf   FAT_File_Pos
    movf    FAT_Dir_Add+1, W
    movwf   FAT_File_Pos+1
    movf    FAT_Dir_Add+2, W
    movwf   FAT_File_Pos+2
    movf    FAT_Dir_Add+3, W
    movwf   FAT_File_Pos+3
       
      
fofw003    
    ; read the sector
    movf    FAT_File_Pos+3, W
    movwf   MMCOp1 
    movf    FAT_File_Pos+2, W
    movwf   MMCOp2
    movf    FAT_File_Pos+1, W
    movwf   MMCOp3 
    movf    FAT_File_Pos, W
    movwf   MMCOp4 
    call    MMCReadBlock
    
    clrf    FAT_Dir_Ind     ; This will count the entries in each sector, up to 16 of them
    lfsr    FSR0, BLOCK_BUFFER_ADDRESS
fofw004    
    
    movf    INDF0, W
    btfsc   STATUS, Z
    goto    fofw005         ; We have found an empty directory slot, which we will fill
    
    incf    FAT_Dir_Ind, F
    movlw   D'32'
    addwf   FSR0L, F
    movlw   0
    addwfc  FSR0H, F
    
    ; If we have scanned the last in this sector, load the next sector
    ; otherwise just loop back
    movf    FAT_Dir_Ind, W
    sublw   D'16'
    btfss   STATUS, Z   
    goto    fofw004
    
    movlw   0x02
    addwf   FAT_File_Pos+1, F
    movlw   0x00
    addwfc  FAT_File_Pos+2, F
    addwfc  FAT_File_Pos+3, F

    goto    fofw003
    
    ; TODO - checking for end of directory - ie, no space - would be good here
        
fofw005
    ; Fill the directory structure and write it back. 
    ; FSR0 is pointing at the start of the entry, and FAT_File_Pos is the 
    ; correct sector to write to.

    lfsr    FSR1, FAT_Filename
    movlw   D'11'
    movwf   FAT_tmp_count

fofw006
    movf    POSTINC1, W
    movwf   POSTINC0
    decfsz  FAT_tmp_count, F
    goto    fofw006
    
    movlw   0x20
    movwf   POSTINC0            ; The attribute byte; Normal file
    
    movlw   D'14'
    movwf   FAT_tmp_count
    
fofw007                         ; zero reserved and time fields
    clrf    POSTINC0
    decfsz  FAT_tmp_count, F
    goto    fofw007           
    
    movf    FAT_FileC, W
    movwf   POSTINC0    
    movf    FAT_FileC+1, W
    movwf   POSTINC0   
     
    clrf    POSTINC0
    clrf    POSTINC0
    
    ; Now write the entry back
    movf    FAT_File_Pos+3, W
    movwf   MMCOp1 
    movf    FAT_File_Pos+2, W
    movwf   MMCOp2
    movf    FAT_File_Pos+1, W
    movwf   MMCOp3 
    movf    FAT_File_Pos, W
    movwf   MMCOp4 
    call    MMCWriteBlock

    clrf    FAT_File_Ind
    clrf    FAT_File_Ind+1
    clrf    FAT_File_Ind+2
    clrf    FAT_File_Ind+3

    clrf    FAT_File_Size
    clrf    FAT_File_Size+1
    clrf    FAT_File_Size+2
    clrf    FAT_File_Size+3
    
    ; Finally, set FAT_File_Pos to the first cluster we will write to.
       
    clrf    FAT_Tmp_Long, 1
    clrf    FAT_Tmp_Long+1, 1
    clrf    FAT_Tmp_Long+2, 1
    clrf    FAT_Tmp_Long+3, 1
    
    movlw   0x02
    subwf   FAT_FileC, F
    movlw   0x00
    subwfb  FAT_FileC+1, F
    
    ; now, make FAT_Tmp_Long equal 512 * the number of sectors per cluster
    movf    FAT_ClusterS, W
    movwf   tmpCount
fofw008    
    movlw   0x02
    addwf   FAT_Tmp_Long+1, F, 1
    movlw   0x00
    addwfc  FAT_Tmp_Long+2, F, 1
    addwfc  FAT_Tmp_Long+3, F, 1
    decfsz  tmpCount, F
    goto    fofw008
   
    clrf    FAT_File_Pos
    clrf    FAT_File_Pos+1
    clrf    FAT_File_Pos+2
    clrf    FAT_File_Pos+3
    
    ; now multiply that by FAT_FileC, if it isn't zero
    
    movf    FAT_FileC, W
    movwf   FAT_tmp_count
    iorwf   FAT_FileC+1, W
    btfsc   STATUS, Z
    goto    fofw009a   
    movf    FAT_FileC+1, W
    movwf   FAT_tmp_count+1
    incf    FAT_tmp_count+1, F
    
fofw009
    movf    FAT_Tmp_Long, W, 1
    addwf   FAT_File_Pos, F
    movf    FAT_Tmp_Long+1, W, 1
    addwfc  FAT_File_Pos+1, F
    movf    FAT_Tmp_Long+2, W, 1
    addwfc  FAT_File_Pos+2, F
    movf    FAT_Tmp_Long+3, W, 1
    addwfc  FAT_File_Pos+3, F
    
    decfsz  FAT_tmp_count, F
    goto    fofw009
    decfsz  FAT_tmp_count+1, F
    goto    fofw009    
            
    ; Then add in the data start offset
fofw009a       
    movf    FAT_Data_Add, W
    addwf   FAT_File_Pos, F
    movf    FAT_Data_Add+1, W
    addwfc  FAT_File_Pos+1, F
    movf    FAT_Data_Add+2, W
    addwfc  FAT_File_Pos+2, F
    movf    FAT_Data_Add+3, W
    addwfc  FAT_File_Pos+3, F        

    bsf     FAT_flags1, flags1_FAT_FILE_IS_OPEN
        
    ; FAT_FileC had been decremented by two; put it back
    movlw   0x02
    addwf   FAT_FileC, F
    movlw   0x00
    addwfc  FAT_FileC+1, F
    
    ; Signal success
    bcf     STATUS, C
 
    return

    
    
;*****************************************************************************        
;
;   Function :  FAT16CloseFileWrite
;               Closes the open file after writing. 
;               If the file has been written to, it updates
;               the FAT tables.
;
;   Input:      None.
;
;   Output:     Various variables updated. Sector buffer over-written.
;
;               
;
;*****************************************************************************        
FAT16CloseFileWrite
    ;
    ; Update the clusters, if we have used any.
    ;
    
    clrf    FAT_Tmp_Long, 1
    clrf    FAT_Tmp_Long+1, 1
    clrf    FAT_Tmp_Long+2, 1
    clrf    FAT_Tmp_Long+3, 1

        ; now, make FAT_Tmp_Long equal number of bytes per cluster
    movf    FAT_ClusterS, W
    movwf   tmpCount
fcf001    
    movlw   0x02
    addwf   FAT_Tmp_Long+1, F, 1
    movlw   0x00
    addwfc  FAT_Tmp_Long+2, F, 1
    addwfc  FAT_Tmp_Long+3, F, 1
    decfsz  tmpCount, F
    goto    fcf001     
    
    ; Use FAT_File_Ind as a copy of the file size; we will use
    ; it to track clusters occupied
    
    movf    FAT_File_Size, W
    movwf   FAT_File_Ind
    movf    FAT_File_Size+1, W
    movwf   FAT_File_Ind+1
    movf    FAT_File_Size+2, W
    movwf   FAT_File_Ind+2
    movf    FAT_File_Size+3, W
    movwf   FAT_File_Ind+3
    
fcf002    
    ; Now, have we used more than a single cluster?
    movf    FAT_Tmp_Long, W, 1
    subwf   FAT_File_Ind, F
    movf    FAT_Tmp_Long+1, W, 1
    subwfb  FAT_File_Ind+1, F
    movf    FAT_Tmp_Long+2, W, 1
    subwfb  FAT_File_Ind+2, F
    movf    FAT_Tmp_Long+3, W, 1
    subwfb  FAT_File_Ind+3, F
    
    btfss   STATUS, C
    goto    fcfDirUpdate        ; No - all updates needed have been done
    
    ; Yes, we have used more than a single cluster.
    ; So write the current cluster record in the FAT at 1 more than itself

    ; Save a copy of the current FAT cluster pointer + 1 - we will need it later
    
    clrf    FAT_tmp_count
    movf    FAT_FileC, W
    addlw   0x01
    movwf   FAT_tmp_count    
    movlw   0x00
    addwfc  FAT_FileC+1, W
    movwf   FAT_tmp_count+1    
    
    ; Make MMCOp point to the FAT table at our cluster entry
    
    ; Take the current cluster number.
    ; Divide by 256 to give the sector offset.
    ; Multiply that by 512 to give byte offset from FAT_FAT_Pos
    ; for the sector that holds our cluster pointer.
    
    clrf    MMCOp4
    movf    FAT_FileC + 1, W
    movwf   MMCOp3
    clrf    MMCOp2
    clrf    MMCOp1
    bcf     STATUS, C
    rlcf    MMCOp3, F
    rlcf    MMCOp2, F
    movf    FAT_FAT_Add, W
    addwf   MMCOp4, F
    movf    FAT_FAT_Add+1, W
    addwfc  MMCOp3, F
    movf    FAT_FAT_Add+2, W
    addwfc  MMCOp2, F
    movf    FAT_FAT_Add+3, W
    addwfc  MMCOp1, F
    
    ; Get that sector    
    call    MMCReadBlock

    ; Now, the two bytes that are our cluster pointer are 
    ; the lower 8 bits of FAT_FileC, * 2
    clrf    FAT_FileC +1
    bcf     STATUS, C
    rlcf    FAT_FileC, F
    rlcf    FAT_FileC+1, F
   
    lfsr    FSR0, BLOCK_BUFFER_ADDRESS
    movf    FAT_FileC, W
    addwf   FSR0L, F
    movf    FAT_FileC+1, W
    addwfc  FSR0H, F    
    
    ; FSR0 Now points to the entry for FAT_FileC, 
    ; Write a value of one more than the original FAT_FileC in there, and store
    ; this new value in FAT_FileC
    movf    FAT_tmp_count, W
    movwf   POSTINC0
    movwf   FAT_FileC
    movf    FAT_tmp_count+1, W
    movwf   POSTINC0    
    movwf   FAT_FileC+1
    
    
    ; write it. The address variable are the same as when we did the read
    call    MMCWriteBlock
    
    goto    fcf002    
    
fcfDirUpdate      
    ; Mark the current cluster record - the last -  as FFFF
    ; Similar code to above
       
    clrf    MMCOp4
    movf    FAT_FileC + 1, W
    movwf   MMCOp3
    clrf    MMCOp2
    clrf    MMCOp1
    bcf     STATUS, C
    rlcf    MMCOp3, F
    rlcf    MMCOp2, F
    movf    FAT_FAT_Add, W
    addwf   MMCOp4, F
    movf    FAT_FAT_Add+1, W
    addwfc  MMCOp3, F
    movf    FAT_FAT_Add+2, W
    addwfc  MMCOp2, F
    movf    FAT_FAT_Add+3, W
    addwfc  MMCOp1, F
    
    ; Get that sector    
    call    MMCReadBlock

    ; Now, the two bytes that are our cluster pointer are 
    ; the lower 8 bits of FAT_FileC, * 2
    clrf    FAT_FileC +1
    bcf     STATUS, C
    rlcf    FAT_FileC, F
    rlcf    FAT_FileC+1, F
   
    lfsr    FSR0, BLOCK_BUFFER_ADDRESS
    movf    FAT_FileC, W
    addwf   FSR0L, F
    movf    FAT_FileC+1, W
    addwfc  FSR0H, F    
    
    ; FSR0 Now points to the entry for FAT_FileC, 
    ; Write an 0xFFFF, indicating the cluster is used, and EOF.
    movlw   0xFF
    movwf   POSTINC0
    movwf   POSTINC0    
    
    ; write it. The address variable are the same as when we did the read
    call    MMCWriteBlock    
        
    ; Update the file size in the directory
  
    movf    FAT_Dir_Add, W
    movwf   FAT_Dir_Pos
    movf    FAT_Dir_Add+1, W
    movwf   FAT_Dir_Pos+1
    movf    FAT_Dir_Add+2, W
    movwf   FAT_Dir_Pos+2
    movf    FAT_Dir_Add+3, W
    movwf   FAT_Dir_Pos+3
    
    clrf    FAT_Dir_Sec     ; counts 0 .. 31 - counts sectors

fcf003    
    clrf    FAT_Dir_Ind     ; counts 0.. 15 - indexes into a dir sector
    
    ; Read the sector holding the directory 

    movf    FAT_Dir_Pos+3, W
    movwf   MMCOp1 
    movf    FAT_Dir_Pos+2, W
    movwf   MMCOp2
    movf    FAT_Dir_Pos+1, W
    movwf   MMCOp3 
    movf    FAT_Dir_Pos, W
    movwf   MMCOp4 
    call    MMCReadBlock

fcf004        
    ; Point FSR0 to the directory entry of interest.
    clrf    FSR0H
    swapf   FAT_Dir_Ind, W
    movwf   FSR0L
    swapf   FAT_Dir_Ind, W
    addwf   FSR0L, F   
    movlw   LOW BLOCK_BUFFER_ADDRESS  
    addwf   FSR0L, F
    movlw   HIGH BLOCK_BUFFER_ADDRESS      
    addwfc  FSR0H, F            
    
    ; Now compare the filename to see if it is the same
    
    lfsr    FSR1, FAT_Filename
    movlw   D'11'
    movwf   tmpCount

fcf005    
    movf    POSTINC0, W
    subwf   POSTINC1, W
    btfss   STATUS, Z
    goto    fcfNextDir
    
    decfsz  tmpCount, F
    goto    fcf005
    
    ; They are the same!
    ; Write the file size in, save it, and exit
    
    clrf    FSR0H
    swapf   FAT_Dir_Ind, W
    movwf   FSR0L
    swapf   FAT_Dir_Ind, W
    addwf   FSR0L, F   
    movlw   LOW BLOCK_BUFFER_ADDRESS  + 0x1c
    addwf   FSR0L, F
    movlw   HIGH BLOCK_BUFFER_ADDRESS      
    addwfc  FSR0H, F

    movf    FAT_File_Size, W
    movwf   POSTINC0
    movf    FAT_File_Size+1, W
    movwf   POSTINC0
    movf    FAT_File_Size+2, W
    movwf   POSTINC0
    movf    FAT_File_Size+3, W
    movwf   POSTINC0
        
    call    MMCWriteBlock
       
    goto    fcfExit    
        
        
fcfNextDir


    incf    FAT_Dir_Ind, F
    movlw   D'16'
    subwf   FAT_Dir_Ind, W
    btfss   STATUS, Z
    goto    fcf004

    clrf    FAT_Dir_Ind
    incf    FAT_Dir_Sec, F

    movlw   0x02
    addwf   FAT_Dir_Pos+1, F
    movlw   0x00
    addwfc  FAT_Dir_Pos+2, F
    addwfc  FAT_Dir_Pos+3, F

    goto    fcf003

fcfExit    
    bcf     FAT_flags1, flags1_FAT_FILE_IS_OPEN
    return



;*****************************************************************************        
;
;   Function :  FAT16WriteFile
;               Writes to the currently open file.
;               The write is fast; you must close the file to properly update
;               the FAT tables
;
;   Input:      None.
;
;   Output:     Various variables updated. Sector buffer over-written.
;
;               
;
;*****************************************************************************        
FAT16WriteFile
    movf    FAT_File_Pos+3, W
    movwf   MMCOp1 
    movf    FAT_File_Pos+2, W
    movwf   MMCOp2
    movf    FAT_File_Pos+1, W
    movwf   MMCOp3 
    movf    FAT_File_Pos, W
    movwf   MMCOp4 
    call    MMCWriteBlock
    
    ; update the write pointer.
    
    movlw   0x02
    addwf   FAT_File_Pos+1, F
    movlw   0x00
    addwfc  FAT_File_Pos+2, F
    addwfc  FAT_File_Pos+3, F
    
    ; update the size pointer; this will help us work out
    ; writing to the FAT tables at the file close

    movlw   0x02
    addwf   FAT_File_Size+1, F
    movlw   0x00
    addwfc  FAT_File_Size+2, F
    addwfc  FAT_File_Size+3, F
        
    return




;*****************************************************************************        
;
;   Function :  FAT16ReadFile
;               Reads data from the file. 512 bytes are read each
;               time.
;
;   Input:      None
;
;   Output:     Various variables updated. Sector buffer written to.
;
;               NOTE: This routine could be improved by detecting the
;               last sector in the file, rather than relying on the 
;               caller detecting EOF.
;
;               
;*****************************************************************************        
FAT16ReadFile      
    
    ; Read Sector. Add 512 to FAT_File_Ind. Add 1 to FAT_SecInClust
    ; move to next cluster if needed.
    
    ; Read sector pointed to by FAT_File_Pos.
    ; Update FAT_File_Pos ( note  cluster change affects )


    decfsz  FAT_SecInClust, F
    goto    nextSect
    
    ; 
    ; Move to next cluster
    
    movlw   0x08 + 1
    movwf   FAT_SecInClust
    
    ; Take the current cluster number.
    ; Divide by 256 to give the sector offset.
    ; Multiply that by 512 to give byte offset from FAT_FAT_Pos
    ; for the sector that holds our cluster pointer.
    
    clrf    MMCOp4
    movf    FAT_FileC + 1, W
    movwf   MMCOp3
    clrf    MMCOp2
    clrf    MMCOp1
    bcf     STATUS, C
    rlcf    MMCOp3, F
    rlcf    MMCOp2, F
    movf    FAT_FAT_Add, W
    addwf   MMCOp4, F
    movf    FAT_FAT_Add+1, W
    addwfc  MMCOp3, F
    movf    FAT_FAT_Add+2, W
    addwfc  MMCOp2, F
    movf    FAT_FAT_Add+3, W
    addwfc  MMCOp1, F
    
    ; Get that sector    
    call    MMCReadBlock
    
    ; Now, the two bytes that are our cluster pointer are 
    ; the lower 8 bits of FAT_FileC, * 2
    clrf    FAT_FileC +1
    bcf     STATUS, C
    rlcf    FAT_FileC, F
    rlcf    FAT_FileC+1, F
   
    lfsr    FSR0, BLOCK_BUFFER_ADDRESS
    movf    FAT_FileC, W
    addwf   FSR0L, F
    movf    FAT_FileC+1, W
    addwfc  FSR0H, F
    
    movf    POSTINC0, W
    movwf   FAT_FileC
    movf    POSTINC0, W
    movwf   FAT_FileC+1
    
    ; Now we know the cluster, go get it.
    clrf    FAT_Tmp_Long, 1
    clrf    FAT_Tmp_Long+1, 1
    clrf    FAT_Tmp_Long+2, 1
    clrf    FAT_Tmp_Long+3, 1
    
    movlw   0x02
    subwf   FAT_FileC, F
    movlw   0x00
    subwfb  FAT_FileC+1, F
    
    ; now, make FAT_Tmp_Long equal 512 * the number of sectors per cluster
    movf    FAT_ClusterS, W
    movwf   tmpCount
frf003    
    movlw   0x02
    addwf   FAT_Tmp_Long+1, F, 1
    movlw   0x00
    addwfc  FAT_Tmp_Long+2, F, 1
    addwfc  FAT_Tmp_Long+3, F, 1
    decfsz  tmpCount, F
    goto    frf003
   
    clrf    FAT_File_Pos
    clrf    FAT_File_Pos+1
    clrf    FAT_File_Pos+2
    clrf    FAT_File_Pos+3
    
    ; now multiply that by FAT_FileC, if it isn't zero
    
    movf    FAT_FileC, W
    movwf   FAT_tmp_count
    iorwf   FAT_FileC+1, W
    btfsc   STATUS, Z
    goto    frf004a 
    movf    FAT_FileC+1, W
    movwf   FAT_tmp_count+1
    incf    FAT_tmp_count+1, F

    
frf004
    movf    FAT_Tmp_Long, W, 1
    addwf   FAT_File_Pos, F
    movf    FAT_Tmp_Long+1, W, 1
    addwfc  FAT_File_Pos+1, F
    movf    FAT_Tmp_Long+2, W, 1
    addwfc  FAT_File_Pos+2, F
    movf    FAT_Tmp_Long+3, W, 1
    addwfc  FAT_File_Pos+3, F
    
    decfsz  FAT_tmp_count, F
    goto    frf004
    decfsz  FAT_tmp_count+1, F
    goto    frf004    
            
    ; Then add in the data start offset
frf004a       
    movf    FAT_Data_Add, W
    addwf   FAT_File_Pos, F
    movf    FAT_Data_Add+1, W
    addwfc  FAT_File_Pos+1, F
    movf    FAT_Data_Add+2, W
    addwfc  FAT_File_Pos+2, F
    movf    FAT_Data_Add+3, W
    addwfc  FAT_File_Pos+3, F

nextSect    
    movf    FAT_File_Pos+3, W
    movwf   MMCOp1 
    movf    FAT_File_Pos+2, W
    movwf   MMCOp2
    movf    FAT_File_Pos+1, W
    movwf   MMCOp3 
    movf    FAT_File_Pos, W
    movwf   MMCOp4 
    call    MMCReadBlock
    
    movlw   0x02
    addwf   FAT_File_Ind + 1, F
    movlw   0x00
    addwfc  FAT_File_Ind + 2, F
    addwfc  FAT_File_Ind + 3, F

    movlw   0x02
    addwf   FAT_File_Pos + 1, F
    movlw   0x00
    addwfc  FAT_File_Pos + 2, F
    addwfc  FAT_File_Pos + 3, F
    
    return



;*****************************************************************************        
;
;   Function :  FAT16StartDirList
;               Sets up some variables in advance, and sets the directory listing
;               state to 'beginning'. Call FAT16NextDirList to get the details 
;               of the next file in the list. Call FAT16NextDirList repeatedly
;               to get further file details. Call FAT16StartDirList to go back 
;               to the beginning. You may do this while a file is open, 
;               but note that the sector buffer is over-written.
;
;
;   Input:      None.
;
;   Output:     Various variables updated. Sector buffer over-written.
;
;               
;*****************************************************************************        
FAT16StartDirList
    ; Just set the directory listing variables to the first ( ie, last )
    ; position in the directory. Subsequent calls work backwards.
    movf    FAT_Dir_Add, W
    movwf   FAT_Dir_Pos
    movf    FAT_Dir_Add+1, W
    movwf   FAT_Dir_Pos+1
    movf    FAT_Dir_Add+2, W
    movwf   FAT_Dir_Pos+2
    movf    FAT_Dir_Add+3, W
    movwf   FAT_Dir_Pos+3
    
    clrf    FAT_Dir_Sec     ; counts 0 .. 31 - counts sectors
    clrf    FAT_Dir_Ind     ; counts 0.. 15 - indexes into a dir sector

    return    
    
    
    
;*****************************************************************************        
;
;   Function :  FAT16NextDirList
;               Reads the details of the next file in the directory list.
;               See FAT16StartDirList.
;
;
;   Input:      None.
;
;   Output:     Various variables updated. Sector buffer over-written.
;               Carry set when no more entries present
;
;               Note: uses FSR0, FSR1
;               
;*****************************************************************************        
FAT16NextDirList

    ; Are we already at the end of the list?
    ; If yes, return CARRY SET
    movlw   D'32'
    subwf   FAT_Dir_Sec, W
    btfss   STATUS, Z
    goto    fnd001

fndEnd        
    bsf     STATUS, C
    return
    
fnd001    
    ; No. Read the appropriate sector, get the relevant 32 byte entry,
    ; update the index variables. CLEAR CARRY

    movf    FAT_Dir_Pos+3, W
    movwf   MMCOp1 
    movf    FAT_Dir_Pos+2, W
    movwf   MMCOp2
    movf    FAT_Dir_Pos+1, W
    movwf   MMCOp3 
    movf    FAT_Dir_Pos, W
    movwf   MMCOp4 
    call    MMCReadBlock
    
    ; now copy just the important 32 bytes
    clrf    FSR0H
    swapf   FAT_Dir_Ind, W
    movwf   FSR0L
    swapf   FAT_Dir_Ind, W
    addwf   FSR0L, F   
    movlw   LOW BLOCK_BUFFER_ADDRESS  
    addwf   FSR0L, F
    movlw   HIGH BLOCK_BUFFER_ADDRESS      
    addwfc  FSR0H, F
    
    lfsr    FSR1, FAT_Dir_Entry    
    
    movlw   D'32'
    movwf   tmpCount
    
fnd002
    movf    POSTINC0, W
    movwf   POSTINC1

    decfsz  tmpCount, F
    goto    fnd002    
    
    ; now, update the two variables, and add 512 to the FAT_Dir_Pos
    ; variable if needed.
    
    incf    FAT_Dir_Ind, F
    movlw   D'16'
    subwf   FAT_Dir_Ind, W
    btfss   STATUS, Z
    goto    fndExit

    clrf    FAT_Dir_Ind
    incf    FAT_Dir_Sec, F

    movlw   0x02
    addwf   FAT_Dir_Pos+1, F
    movlw   0x00
    addwfc  FAT_Dir_Pos+2, F
    addwfc  FAT_Dir_Pos+3, F

fndExit
    ; Check to see if this has any entries in it. IF not, move to the
    ; next one.
    ; The end of the list is determined by 0x00
    ; An unused entry is 0xE5
    
    movf    FAT_Dir_Entry, W
    btfsc   STATUS, Z
    goto    fndEnd
    
    movf    FAT_Dir_Entry, W
    sublw   0xE5
    btfsc   STATUS, Z
    goto    FAT16NextDirList

    bcf     STATUS, C
    return

