;*****************************************************************************        
;
;   Module:     list1.asm
;               
;   Author:     Mike Hibbett 
;                                                                  
;   Version:    0.1 29/10/07                                                  
;
;               Demonstrates a very simple user interface
;               This code can be turned into a .hex file
;               by simply opening it in MPLAB and selecting
;               'Project' then 'QuickBuild List1.asm'
;
;*****************************************************************************        

    list p=18F2420    

	; Suppress the display of informational warning messages
    errorlevel -302
    errorlevel -306

    #include <p18f2420.inc>
    


;*****************************************************************************        
;
;   Function :  Reset vector
;               Hardware entry point to the code
;               
;
;   Input:      None.
;
;   Output:     N/A
;
;*****************************************************************************        
    
    ORG    0

	call	initHardware
	
main	

	call	lockDoor
	call	displayLocked
	call	getKey
	call	displayUnlocked
	call	unlockDoor
	call	openDelay
	
    goto    main





;*****************************************************************************        
;
;   Function : initHardware
;              Configures the speed of the on-board oscillator,
;              sets up the I/O ports and initialises the LCD
;
;   Input:     None.
;
;*****************************************************************************        
initHardware
	
	; Set oscillator to 8MHz.
	movlw	0x73
	movwf	OSCCON
	
    ; Allow PORTB pins to be set to digital I/O
    movlw   ADC_OFF_VAL
    movwf   ADCON1     

	; Make PORTB output, for driving LCD    
	movlw	0xFF
	movwf	PORTB
    clrf    TRISB
	
    ;   First, delay for 1/2s for things to settle
    movlw   D'50'
    call    UIWait10ms
        
    ; Make port RC inputs, for the keys
    movlw	0xFF
    movwf	TRISC


    ; Now setup the LCD into 4-bit mode, and clear the display
    call    DspPutCmd4Bit

	; Init the display: 4 bit, 2 line, small font
    movlw   D'4'
    call    UIWait10ms
    movlw   0x28
    call    DspPutCmd            
    
	; repeat, since first time doesn't always work        
    movlw   D'4'
    call    UIWait10ms
    movlw   0x28
    call    DspPutCmd       

	; turn the display on, and move to the start, with no cursor
    movlw   0x00C
    call    DspPutCmd           
    call    DspClear

	return    
 


;*****************************************************************************        
;
;   Function : lockDoor
;              Turn the door lock solenoid on, locking the door.
;              (Just a stub routine in this example - it is not important)
;
;   Input:     None.
;
;*****************************************************************************        
lockDoor
	return
	
	
	
;*****************************************************************************        
;
;   Function : unlockDoor
;              Turn the door lock solenoid off, unlocking the door.
;              (Just a stub routine in this example - it is not important)
;
;   Input:     None.
;
;*****************************************************************************        
unlockDoor
	return
	
	
	
;*****************************************************************************        
;
;   Function : getKey
;              Waits for a key to be pressed, and then released
;              The key is on RC0
;
;   Input:     None.
;
;*****************************************************************************        
getKey
	; wait for key to be pressed ( a zero )
	btfsc	PORTC, 0
	goto	getKey
	; delay, to check to see if it is really pressed
    movlw   D'2'
    call    UIWait10ms
	btfsc	PORTC, 0
	goto	getKey

	; now wait for it to be released
waitRelease
	btfss	PORTC, 0
	goto	waitRelease
	; delay, to check to see if it is really pressed
    movlw   D'2'
    call    UIWait10ms
	btfss	PORTC, 0
	goto	waitRelease
	
	return



;*****************************************************************************        
;
;   Function : openDelay
;              Waits for a few seconds ( 5s, to allow the door to be pulled 
;              open )
;
;   Input:     None.
;
;*****************************************************************************        
openDelay
    movlw   D'250'
    call    UIWait10ms
    movlw   D'250'
    call    UIWait10ms
	return



;*****************************************************************************        
;
;   Function : displayLocked
;              Displays the message "Door Locked" on the LCD
;
;   Input:     None.
;
;*****************************************************************************        
displayLocked
    call    DspClear
    movlw   HIGH STRLocked
    movwf   TBLPTRH
    movlw   LOW STRLocked
    movwf   TBLPTRL
    call    UIDspStr
	return
	
	
	
;*****************************************************************************        
;
;   Function : displayUnlocked
;              Displays the message "Door Unlocked" on the LCD
;
;   Input:     None.
;
;*****************************************************************************        
displayUnlocked
    call    DspClear
    movlw   HIGH STRUnlocked
    movwf   TBLPTRH
    movlw   LOW STRUnlocked
    movwf   TBLPTRL
    call    UIDspStr
	return



;*****************************************************************************        
;
;   Function :  UIDspStr
;               Write a string to the display at the current cursor position.
;               This does not handle line wrapping
;
;   Input:      TBLPTRH/L setup with string address.
;
;   Output:     None
;
;*****************************************************************************        
UIDspStr
UID001
    tblrd*+				; Read a byte from flash memory
    movf    TABLAT, W
    bz      UID000
    call    DspPutChar
    bra     UID001    
    
UID000
    return

STRLocked
	DB "Door Locked",0
STRUnlocked
	DB "Door Unlocked",0

 
;*****************************************************************************        
;
;   Function : UIWait10ms
;              delays for a multiple of 10ms
;
;   Input:     multiple in W
;
;*****************************************************************************        
UIWait10ms
    movwf   delay3 
    
d1ms001
    movlw   D'200'
    call    UIWait50us
    decfsz  delay3, F 
    bra     d1ms001
    return



;*****************************************************************************        
;
;   Function : UIWait50us
;              delays for a multiple of 50us
;
;   Input:     multiple in W
;              cycles are 2 + 1 + (delay2)*( 1+1 + ((d-1)*3 + 2) + 3)  + 3
;              Need 500 cycles at 40MHz : delay1 = 165
;              Need 100 cycles at 8MHz: delay1 = 32
;                         
;
;*****************************************************************************        
UIWait50us             ; 2     ( the call itself )
    movwf   delay2      ; 1  

d1us002
    movlw   D'32'      ; 1          
    movwf   delay1      ; 1          
    
d1us001      
    decfsz  delay1, F   ; 1 or 2        
    bra     d1us001     ; 2               
    
    decfsz  delay2, F   ; 1 or 2
    bra     d1us002     ; 2
    return              ; 2        



;*****************************************************************************        
;
;   Function :  LCDWriteDelay
;               Provides a small delay when driving the LCD pins
;              
;   Input:      None
;
;   Output:     None
;
;*****************************************************************************        
LCDWriteDelay
    nop
    nop
    return  
    
    
    
;*****************************************************************************        
;
;   Function :  dspPutCmd4Bit
;               Puts the LCD into 4 bit mode from 8 bit mode
;              
;   Input:      Command to write is in W
;
;   Output:     None
;
;*****************************************************************************        
DspPutCmd4Bit
    bcf     CONTROL, LCD_RW 
    call    LCDWriteDelay
    bcf     CONTROL, LCD_RS 
    call    LCDWriteDelay
    bsf     CONTROL, LCD_E 
    movlw   0xF0
    andwf   CONTROL, F    
    movlw   0x02
    iorwf   CONTROL, F    
    call    LCDWriteDelay
    bcf     CONTROL, LCD_E             
    call    LCDWriteDelay
    return  



;*****************************************************************************        
;
;   Function :  LCDBusy
;               This call blocks until the busy flag in the LCD is clear
;
;   Input:      None
;
;   Output:     None
;
;*****************************************************************************        
LCDBusy
    movlw   DATA_BUS_INPUT
    movwf   DATA_BUS_TRIS    
    bcf     CONTROL, LCD_RS      ; Set LCD for command mode
    call    LCDWriteDelay
    bsf     CONTROL, LCD_RW      ; Setup to read busy flag
    call    LCDWriteDelay
    bsf     CONTROL, LCD_E       ; LCD E-line High
    call    LCDWriteDelay
    movf    PORTB, W             ; Read busy flag + DDram address
    andlw   0x0F
    movwf   lcdTmp2    
    bcf     CONTROL, LCD_E       ; LCD E-line Low
    call    LCDWriteDelay
    bsf     CONTROL, LCD_E       ; LCD E-line High
    call    LCDWriteDelay
    swapf   PORTB, W             ; Read busy flag + DDram address
    andlw   0xF0
    addwf   lcdTmp2, F    
    bcf     CONTROL, LCD_E       ; LCD E-line Low    
    movf    lcdTmp2, W 
    andlw   0x08                    ; Check Busy flag, High = Busy
    btfss   STATUS, Z 
    bra     LCDBusy    
    bcf     CONTROL, LCD_RW 
    call    LCDWriteDelay
    movlw   DATA_BUS_OUTPUT
    movwf   DATA_BUS_TRIS 
    return
       
       

;*****************************************************************************        
;
;   Function :  DspPutChar
;               This call writes a character to the current cursor position.
;
;   Input:      Character in W
;
;   Output:     None
;
;*****************************************************************************        
DspPutChar
    movwf   lcdTmp               ; Character to be sent is in W
    call    LCDBusy                 ; Wait for LCD to be ready
    bcf     CONTROL, LCD_RW      ; Set LCD in read mode
    call    LCDWriteDelay
    bsf     CONTROL, LCD_RS      ; Set LCD in data mode
    call    LCDWriteDelay
    bsf     CONTROL, LCD_E       ; LCD E-line High
    call    LCDWriteDelay
    movlw   0xF0
    andwf   CONTROL, F    
    swapf   lcdTmp, W 
    andlw   0x0F
    iorwf   CONTROL, F           ; Send data to LCD
    call    LCDWriteDelay
    bcf     CONTROL, LCD_E       ; LCD E-line Low
    call    LCDWriteDelay
    bsf     CONTROL, LCD_E       ; LCD E-line High
    call    LCDWriteDelay
    movlw   0xF0
    andwf   CONTROL, F    
    movf    lcdTmp, W 
    andlw   0x0F
    iorwf   CONTROL, F           ; Send data to LCD
    call    LCDWriteDelay
    bcf     CONTROL, LCD_E       ; LCD E-line Low
    call    LCDWriteDelay
    return
  


;*****************************************************************************        
;
;   Function :  DspPutCmd
;               Sends a command to the LCD. 
;              
;   Input:      Command to write is in W
;
;   Output:     None
;
;*****************************************************************************        
DspPutCmd
    movwf   lcdTmp              
    call    LCDBusy               
    bcf     CONTROL, LCD_RW 
    call    LCDWriteDelay
    bcf     CONTROL, LCD_RS 
    call    LCDWriteDelay
    bsf     CONTROL, LCD_E
    movlw   0xF0
    andwf   CONTROL, F    
    swapf   lcdTmp, W            ; Retrieve command byte, send first nibble
    andlw   0x0F
    iorwf   CONTROL, F 
    call    LCDWriteDelay
    bcf     CONTROL, LCD_E    
    call    LCDWriteDelay
    movlw   0xF0
    andwf   CONTROL, F    
    bsf     CONTROL, LCD_E 
    movf    lcdTmp, W            ; Retrieve command byte, send second nibble
    andlw   0x0F
    iorwf   CONTROL, F 
    call    LCDWriteDelay
    bcf     CONTROL, LCD_E    
    call    LCDWriteDelay
    return  



;*****************************************************************************        
;
;   Function :  DspClear
;               This call clears the display and returns the cursor position
;               to top left.
;
;   Input:      None
;
;   Output:     None
;
;*****************************************************************************        
DspClear
    movlw   0x001
    call    DspPutCmd
    return
    
    
    

; symbols defined to make the code more readable
; LCD Port connections
LCD_E                   EQU 4
LCD_RW                  EQU 6
LCD_RS                  EQU 5

CONTROL                 EQU LATB
DATA_BUS                EQU LATB
DATA_BUS_TRIS           EQU TRISB
DATA_BUS_INPUT          EQU 0x0F
DATA_BUS_OUTPUT         EQU 0x00

ADC_OFF_VAL             EQU 0x0F
    
	; Give some defaults settings for the config registers 
	CONFIG OSC = INTIO7, WDT = OFF, PBADEN = OFF, LVP=OFF, MCLRE = OFF

; Declare the locations for variables we use in this
; program
lcdTmp 		EQU 0x00
lcdTmp2     EQU 0x01
delay1      EQU 0x02
delay2      EQU 0x03
delay3      EQU 0x04
    
    END                ; End of program
    