;*****************************************************************************        
;
;   Module:     i2c.asm
;               
;   Author:     Mike Hibbett 
;                                                                  
;   Version:    1.0 10/12/04                                                  
;
;               Code for accessing the i2c eeprom (24lc512)
;               This code also works with smaller eeprom, eg 24lc64
;
;*****************************************************************************        



; EEPROM Format  Word values are MSB then LSB
;
;  Address  |	Number of bytes  |	Use
;  ---------+---------------------+---------------------------------------------
;  0x0000	    1					Hardware type ( 0 == LDR, 1 == 3 inputs )
;  0x0001	    2					Wav file 1 Start Address in eeprom
;  0x0003       2                   Wav File 1 Length
;  0x0005		2					Wav file 2 Start Address in eeprom
;  0x0007       2                   Wav File 2 Length
;  0x0009		2					Wav file 3 Start Address in eeprom
;  0x000b       2                   Wav File 3 Length
;  0x000d        .....				Wav file data goes in here




;*****************************************************************************        
;
;   Function :  eepHWInit
;               configures the I2C peripherial hardware
;
;   Input:      None
;
;   Output:     None
;
;*****************************************************************************        
eepHWInit
    bsf     STATUS, RP0
    bcf     I2C_TRIS, SDA
    bcf     I2C_TRIS, SCL
    bcf     STATUS, RP0
    
    ; Drive the pins low; We will use port direction to make 
    ; pin high
    bcf     I2C_PORT,SDA
    bcf     I2C_PORT,SCL
    
    ; Raise the pins to the bus default state (high)
    bsf     STATUS, RP0
    bsf     I2C_TRIS, SDA
    bsf     I2C_TRIS, SCL
    bcf     STATUS, RP0

    movlw   0x01
    call    uiWait10ms
    return
    


sdaHigh
    bsf     STATUS, RP0                
    bsf     I2C_TRIS, SDA                
    bcf     STATUS, RP0                 
    return
    
sdaLow
    bsf     STATUS, RP0                 
    bcf     I2C_TRIS, SDA                   
    bcf     STATUS, RP0                 
    return

sclHigh
    bsf     STATUS, RP0                
    bsf     I2C_TRIS, SCL
    bcf     STATUS, RP0                 
    nop
    return
    
sclLow
    bsf     STATUS, RP0                 
    bcf     I2C_TRIS, SCL                   
    bcf     STATUS, RP0                 
    return

    
;*****************************************************************************        
;
;   Function :  tx
;               Sends an 8 bit byte over the I2C bus 
;
;   Input:      8 bit byte in eepBuff
;
;   Output:     None
;
;*****************************************************************************        
tx
    movlw   0x01
    movwf   bitCount
    
tx001
    btfsc   eepBuff, 7
    call    sdaHigh     
    btfss   eepBuff, 7
    call    sdaLow   
    
    call    sclHigh
    call    sclLow
    
    rlf     eepBuff, F
    rlf     bitCount, F
    btfss   STATUS,C  
    goto    tx001
    
    return
    
    

;*****************************************************************************        
;
;   Function :  rx
;               receives an 8 bit byte over the I2C bus
;
;   Input:      None
;
;   Output:     Received byte in eepBuff
;
;*****************************************************************************        
rx
    movlw   0x01
    movwf   bitCount
    
rx001
    call    sdaHigh     
    call    sclHigh
    movfw   I2C_PORT
    andlw   SDA_BIT

    bcf     eepBuff, 0
    btfss   STATUS,Z
    bsf     eepBuff, 0

    call    sclLow
    rlf     bitCount,F
    
    btfsc   STATUS,C
    return
    
    rlf     eepBuff,F
    goto    rx001
        


;*****************************************************************************        
;
;   Function :  start
;               Generate a start condition on the I2C bus
;
;   Input:      None
;
;   Output:     None
;
;*****************************************************************************        
start
    call    sdaHigh     
    call    sclHigh
    call    sdaLow     
    call    sclLow
    return



;*****************************************************************************        
;
;   Function :  stop
;               generate an I2C stop condition on the bus
;
;   Input:      None
;
;   Output:     None
;
;*****************************************************************************        
stop
    call    sdaLow     
    call    sclHigh
    call    sdaHigh     
    return




getAck
    call    sdaHigh     
    call    sclHigh
ga001
    movfw   I2C_PORT
    andlw   SDA_BIT
    btfss   STATUS,Z
    goto    ga001
    call    sclLow
    return
    
setAck
    call    sdaLow     
    call    sclHigh
    call    sclLow
    return
    

nAck
    call    sdaHigh     
    call    sclHigh
    call    sclLow
    return
    
    
; eepAddH, eepAddL, eepData
eepWrite
    call    start
    movlw   B'10100000'   
    movwf   eepBuff
    call    tx
    call    getAck
    movfw   eepAddH
    movwf   eepBuff
    call    tx
    call    getAck
    movfw   eepAddL
    movwf   eepBuff
    call    tx
    call    getAck
    movfw   eepData
    movwf   eepBuff
    call    tx
    call    getAck
    call    stop
    
    return
    

eepWritePageFirst
    call    start
    movlw   B'10100000'   
    movwf   eepBuff
    call    tx
    call    getAck
    movfw   eepAddH
    movwf   eepBuff
    call    tx
    call    getAck
    movfw   eepAddL
    movwf   eepBuff
    call    tx
    call    getAck
    movfw   eepData
    movwf   eepBuff
    call    tx
    call    getAck
    
    return
    
eepWritePageNext
    movfw   eepData
    movwf   eepBuff
    call    tx
    call    getAck
    
    return

eepWritePageStop
    movfw   eepData
    movwf   eepBuff
    call    tx
    call    getAck
    call    stop
    return


eepRead
    call    start
    movlw   B'10100000'   
    movwf   eepBuff
    call    tx
    call    getAck
    movfw   eepAddH
    movwf   eepBuff
    call    tx
    call    getAck
    movfw   eepAddL
    movwf   eepBuff
    call    tx
    call    getAck

    call    start
    movlw   B'10100001'   
    movwf   eepBuff
    call    tx
    call    getAck
    call    rx    
    call    nAck
    call    stop 
    movfw   eepBuff
    movwf   eepData
    
    return
    
    
eepReadFirst
    call    start
    movlw   B'10100000'   
    movwf   eepBuff
    call    tx
    call    getAck
    movfw   eepAddH
    movwf   eepBuff
    call    tx
    call    getAck
    movfw   eepAddL
    movwf   eepBuff
    call    tx
    call    getAck

    call    start
    movlw   B'10100001'   
    movwf   eepBuff
    call    tx
    call    getAck
    call    rx    
    call    setAck
    movfw   eepBuff
    movwf   eepData

    return
    
eepReadNext
    call    rx    
    call    setAck
    movfw   eepBuff
    movwf   eepData
    return
    
eepReadLast
    call    rx    
    call    nAck
    call    stop 
    movfw   eepBuff
    movwf   eepData
    return